<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\CalculatorController;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\AuthController;
use App\Http\Controllers\Admin\SetupController;
use App\Http\Controllers\ContactController;
use App\Http\Controllers\Admin\FaqController;
use App\Http\Controllers\Admin\ContactSubmissionController;
use App\Http\Controllers\Admin\UserController;
use App\Http\Controllers\Admin\RoleController;
use App\Http\Controllers\Admin\CountryController;
use App\Http\Controllers\Admin\ReportController;
use App\Http\Controllers\Admin\AuditLogController;
use App\Http\Controllers\Admin\SettingController;
use App\Http\Controllers\Admin\HsCodeController;
use App\Http\Controllers\Admin\TaxConfigController;
use App\Http\Controllers\Admin\BenchmarkController;
use App\Http\Controllers\Admin\CountryOverrideController;
use App\Http\Controllers\Admin\CurrencyController;
use App\Http\Controllers\Admin\ExchangeRateController;
use App\Http\Controllers\Admin\TaxVersionController;
use App\Http\Controllers\Admin\CalculationLogController;
use App\Http\Controllers\Admin\TaxCalculatorController;
use App\Http\Controllers\Admin\BulkImportController;
use App\Http\Controllers\Admin\ProfileController;
use App\Http\Controllers\Admin\SystemHealthController;

// Public guest routes
Route::get('/', function () {
    return view('welcome');
})->name('home');

Route::get('/calculator', [CalculatorController::class, 'index'])->name('calculator');
Route::post('/calculator/calculate', [CalculatorController::class, 'calculate'])->name('calculator.calculate');
Route::get('/calculator/results/{id}', [CalculatorController::class, 'results'])->name('calculator.results');

Route::view('/assistance', 'guest.assistance')->name('assistance');
Route::view('/about', 'guest.about')->name('about');

// Contact routes
Route::get('/contact', [ContactController::class, 'index'])->name('contact');
Route::post('/contact/submit', [ContactController::class, 'submit'])->name('contact.submit');

// Test route - remove after setup
Route::get('/test-setup', function () {
    $adminExists = \App\Models\User::where('is_admin', true)->exists();
    
    if ($adminExists) {
        return "✅ Admin exists - should redirect to login<br>
               <a href='" . route('login') . "'>Go to Login</a>";
    } else {
        return "❌ No admin exists - should show setup form<br>
               <a href='" . route('admin.setup') . "'>Go to Setup</a>";
    }
});

// Admin Setup (Handle logic in controller instead of middleware)
Route::prefix('admin')->name('admin.')->group(function () {
    Route::get('/setup', [SetupController::class, 'showSetupForm'])->name('setup');
    Route::post('/setup', [SetupController::class, 'setup'])->name('setup.submit');
});

// Authentication routes
Route::get('/login', [AuthController::class, 'showLoginForm'])->name('login');
Route::post('/login', [AuthController::class, 'login'])->name('login.submit');
Route::post('/logout', [AuthController::class, 'logout'])->name('logout');

// Admin routes (only for authenticated + admin users)
Route::middleware(['auth'])->prefix('admin')->name('admin.')->group(function () {
    // Dashboard
    Route::get('/', [DashboardController::class, 'index'])->name('dashboard');
    
    // Quick seed data route
    Route::post('/seed-data', [DashboardController::class, 'seedData'])->name('seed.data');
    
    // FAQ Management Routes
    Route::resource('faqs', FaqController::class);
    Route::patch('faqs/{faq}/toggle-status', [FaqController::class, 'toggleStatus'])->name('faqs.toggle-status');
    Route::post('faqs/bulk-action', [FaqController::class, 'bulkAction'])->name('faqs.bulk-action');
    
    // Contact Submissions Management
    Route::get('contact-submissions', [ContactSubmissionController::class, 'index'])->name('contact-submissions.index');
    Route::get('contact-submissions/{submission}', [ContactSubmissionController::class, 'show'])->name('contact-submissions.show');
    Route::patch('contact-submissions/{submission}/mark-read', [ContactSubmissionController::class, 'markAsRead'])->name('contact-submissions.mark-read');
    Route::patch('contact-submissions/{submission}/mark-unread', [ContactSubmissionController::class, 'markAsUnread'])->name('contact-submissions.mark-unread');
    Route::post('contact-submissions/{submission}/send-response', [ContactSubmissionController::class, 'sendResponse'])->name('contact-submissions.send-response');
    Route::post('contact-submissions/{submission}/add-note', [ContactSubmissionController::class, 'addNote'])->name('contact-submissions.add-note');
    Route::post('contact-submissions/{submission}/update-status', [ContactSubmissionController::class, 'updateStatus'])->name('contact-submissions.update-status');
    Route::post('contact-submissions/{submission}/assign', [ContactSubmissionController::class, 'assign'])->name('contact-submissions.assign');
    Route::delete('contact-submissions/{submission}', [ContactSubmissionController::class, 'destroy'])->name('contact-submissions.destroy');
    Route::post('contact-submissions/bulk-action', [ContactSubmissionController::class, 'bulkAction'])->name('contact-submissions.bulk-action');

    // User Management
    Route::prefix('users')->name('users.')->group(function () {
        Route::get('/', [UserController::class, 'index'])->name('index');
        Route::get('/create', [UserController::class, 'create'])->name('create');
        Route::post('/', [UserController::class, 'store'])->name('store');
        Route::get('/{user}/edit', [UserController::class, 'edit'])->name('edit');
        Route::put('/{user}', [UserController::class, 'update'])->name('update');
        Route::delete('/{user}', [UserController::class, 'destroy'])->name('destroy');
        Route::post('/{user}/toggle-status', [UserController::class, 'toggleStatus'])->name('toggle-status');
        Route::post('/{user}/reset-password', [UserController::class, 'resetPassword'])->name('reset-password');
    });
    // Role Management
    Route::prefix('roles')->name('roles.')->group(function () {
        Route::get('/', [RoleController::class, 'index'])->name('index');
        Route::get('/create', [RoleController::class, 'create'])->name('create');
        Route::post('/', [RoleController::class, 'store'])->name('store');
        Route::get('/{role}/edit', [RoleController::class, 'edit'])->name('edit');
        Route::put('/{role}', [RoleController::class, 'update'])->name('update');
        Route::delete('/{role}', [RoleController::class, 'destroy'])->name('destroy');
    });

    // Continue with other routes...

        Route::resource('hs-codes', HsCodeController::class);
        Route::get('hs-codes/import', [HsCodeController::class, 'import'])->name('hs-codes.import'); // Fixed route name
        Route::post('hs-codes/bulk-import', [HsCodeController::class, 'bulkImport'])->name('hs-codes.bulk-import');
        Route::get('hs-codes/export', [HsCodeController::class, 'export'])->name('hs-codes.export');
        Route::get('hs-codes/template', [HsCodeController::class, 'downloadTemplate'])->name('hs-codes.template');
 

    // Tax Configurations
    Route::resource('tax-configs', TaxConfigController::class);
    Route::get('tax-configs/{id}/clone', [TaxConfigController::class, 'clone'])->name('tax-configs.clone');
    
    // Benchmarks Management
    Route::resource('benchmarks', BenchmarkController::class);
    Route::post('benchmarks/bulk-import', [BenchmarkController::class, 'bulkImport'])->name('benchmarks.bulk-import');
    
    // Country Management
    Route::resource('countries', CountryController::class);
    
    // Country Duty Overrides
    Route::resource('country-overrides', CountryOverrideController::class);
    Route::get('country-overrides/{id}/clone', [CountryOverrideController::class, 'clone'])->name('country-overrides.clone');
    
    // Currencies & Exchange Rates
    Route::resource('currencies', CurrencyController::class);
    Route::resource('exchange-rates', ExchangeRateController::class);
    Route::post('exchange-rates/quick-update', [ExchangeRateController::class, 'quickUpdate'])->name('exchange-rates.quick-update');
    Route::post('exchange-rates/bulk-update', [ExchangeRateController::class, 'bulkUpdate'])->name('exchange-rates.bulk-update');
    
    // Tax Rule Versions
    Route::resource('tax-versions', TaxVersionController::class);
    Route::post('tax-versions/{id}/activate', [TaxVersionController::class, 'activate'])->name('tax-versions.activate');
    Route::post('tax-versions/{id}/clone', [TaxVersionController::class, 'clone'])->name('tax-versions.clone');
    
    // Calculation Logs
    Route::resource('calculation-logs', CalculationLogController::class);
    Route::get('calculation-logs/{id}/recalculate', [CalculationLogController::class, 'recalculate'])->name('calculation-logs.recalculate');
    Route::post('calculation-logs/bulk-delete', [CalculationLogController::class, 'bulkDelete'])->name('calculation-logs.bulk-delete');
    Route::get('calculation-logs/export/{type}', [CalculationLogController::class, 'export'])->name('calculation-logs.export');
    
    // Tax Calculator (Admin Version)
    Route::get('tax-calculator', [TaxCalculatorController::class, 'index'])->name('tax-calculator');
    Route::post('tax-calculator/calculate', [TaxCalculatorController::class, 'calculate'])->name('tax-calculator.calculate');
    Route::get('tax-calculator/history', [TaxCalculatorController::class, 'history'])->name('tax-calculator.history');
    
    // Bulk Import Interface
    Route::get('bulk-import', [BulkImportController::class, 'index'])->name('bulk-import');
    Route::post('bulk-import/process', [BulkImportController::class, 'process'])->name('bulk-import.process');
    Route::get('bulk-import/template/{type}', [BulkImportController::class, 'downloadTemplate'])->name('bulk-import.template');
    
    // Audit Logs
    Route::get('audit-logs', [AuditLogController::class, 'index'])->name('audit-logs');
    Route::get('audit-logs/{id}', [AuditLogController::class, 'show'])->name('audit-logs.show');
    Route::post('audit-logs/clear', [AuditLogController::class, 'clear'])->name('audit-logs.clear');
    
    // System Settings

    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('/', [SettingController::class, 'index'])->name('index');
        Route::post('/general', [SettingController::class, 'updateGeneral'])->name('general');
        Route::post('/contact', [SettingController::class, 'updateContact'])->name('contact');
        Route::post('/tax', [SettingController::class, 'updateTax'])->name('tax');
        Route::post('/notifications', [SettingController::class, 'updateNotifications'])->name('notifications');
        Route::delete('/remove-logo', [SettingController::class, 'removeLogo'])->name('remove-logo');
    });
    
    // Profile Management

    Route::get('profile', [ProfileController::class, 'index'])->name('profile');
    Route::put('profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::put('profile/password', [ProfileController::class, 'updatePassword'])->name('profile.password');
    
    
// Reports
    Route::prefix('reports')->name('reports.')->group(function () {
        Route::get('/', [ReportController::class, 'index'])->name('index');
        Route::get('/tax-collection', [ReportController::class, 'taxCollection'])->name('tax-collection');
        Route::get('/hs-code-analysis', [ReportController::class, 'hsCodeAnalysis'])->name('hs-code-analysis');
        Route::get('/country-analysis', [ReportController::class, 'countryAnalysis'])->name('country-analysis');
        Route::post('/generate', [ReportController::class, 'generate'])->name('generate');
        Route::get('/export/{type}', [ReportController::class, 'export'])->name('export');
    });
    
   // API Documentation
Route::get('api-docs', function () {
    return view('admin.api-docs');
})->name('api-docs');

// Postman collection export
Route::get('api-docs/export', function () {
    $postmanCollection = [
        'info' => [
            'name' => 'Kenya Import Tax Simulator API',
            'description' => 'API for calculating import taxes in Kenya',
            'schema' => 'https://schema.getpostman.com/json/collection/v2.1.0/collection.json'
        ],
        'item' => [
            [
                'name' => 'Tax Calculation',
                'request' => [
                    'method' => 'GET',
                    'header' => [],
                    'url' => [
                        'raw' => '{{url}}/api/v1/tax-calculate',
                        'host' => ['{{url}}'],
                        'path' => ['api', 'v1', 'tax-calculate'],
                        'query' => [
                            ['key' => 'hs_code', 'value' => '870321', 'description' => 'HS Code'],
                            ['key' => 'cif_value', 'value' => '1500000', 'description' => 'CIF Value in KES'],
                            ['key' => 'country_of_origin', 'value' => 'CN', 'description' => 'Country Code'],
                        ]
                    ]
                ]
            ]
            // Add more endpoints as needed
        ]
    ];

    return response()->json($postmanCollection, 200, [
        'Content-Disposition' => 'attachment; filename="tax-simulator-postman-collection.json"',
        'Content-Type' => 'application/json',
    ]);
})->name('api-docs.export');
    
    // System Health
    Route::prefix('system-health')->name('system-health.')->group(function () {
        Route::get('/', [SystemHealthController::class, 'index'])->name('index');
        Route::post('/clear-cache', [SystemHealthController::class, 'clearCache'])->name('clear-cache');
        Route::post('/run-backup', [SystemHealthController::class, 'runBackup'])->name('run-backup');
        Route::get('/download-backup/{filename}', [SystemHealthController::class, 'downloadBackup'])->name('download-backup');
        Route::delete('/delete-backup/{filename}', [SystemHealthController::class, 'deleteBackup'])->name('delete-backup');
    });
});

// API Routes (for external integrations)
Route::prefix('api')->name('api.')->group(function () {
    Route::get('v1/tax-calculate', [App\Http\Controllers\Api\V1\TaxCalculatorApiController::class, 'calculate'])->name('v1.tax-calculate');
    Route::get('v1/hs-codes', [App\Http\Controllers\Api\V1\HsCodeApiController::class, 'index'])->name('v1.hs-codes');
    Route::get('v1/hs-codes/{code}', [App\Http\Controllers\Api\V1\HsCodeApiController::class, 'show'])->name('v1.hs-codes.show');
    Route::get('v1/countries', [App\Http\Controllers\Api\V1\CountryApiController::class, 'index'])->name('v1.countries');
    Route::get('v1/currencies', [App\Http\Controllers\Api\V1\CurrencyApiController::class, 'index'])->name('v1.currencies');
    Route::get('v1/exchange-rates', [App\Http\Controllers\Api\V1\ExchangeRateApiController::class, 'index'])->name('v1.exchange-rates');
    
    // Authenticated API routes (for partners with API keys)
    Route::middleware(['auth:api'])->prefix('v1')->group(function () {
        Route::post('bulk-calculate', [App\Http\Controllers\Api\V1\BulkCalculationApiController::class, 'calculate'])->name('v1.bulk-calculate');
        Route::get('calculation-history', [App\Http\Controllers\Api\V1\CalculationHistoryApiController::class, 'index'])->name('v1.calculation-history');
    });
});

// Simple fallback route
Route::fallback(function () {
    return response('<div style="text-align: center; padding: 50px; font-family: Arial, sans-serif;">
        <h1 style="color: #666;">404 - Page Not Found</h1>
        <p style="color: #888; margin-bottom: 30px;">The page you are looking for does not exist.</p>
        <a href="' . url('/') . '" style="background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;">Go to Homepage</a>
    </div>', 404);
});