@extends('layouts.admin')

@section('title', 'Tax Calculator - Admin')

@section('content')
<div class="container-fluid px-4">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">
                            <i class="fas fa-calculator me-2"></i>Tax Calculator - Admin
                        </h4>
                        <div class="btn-group">
                            <button type="button" class="btn btn-light btn-sm" data-bs-toggle="modal" data-bs-target="#quickGuideModal">
                                <i class="fas fa-question-circle me-1"></i>Guide
                            </button>
                            <a href="{{ route('admin.tax-calculator.history') }}" class="btn btn-light btn-sm">
                                <i class="fas fa-history me-1"></i>History
                            </a>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <form id="adminTaxCalculatorForm" action="{{ route('admin.tax-calculator.calculate') }}" method="POST">
                        @csrf
                        
                        <div class="row">
                            <!-- Left Column - Input Form -->
                            <div class="col-md-6">
                                <div class="card border-0 shadow-sm mb-4">
                                    <div class="card-header bg-light">
                                        <h5 class="mb-0"><i class="fas fa-edit me-2"></i>Calculation Inputs</h5>
                                    </div>
                                    <div class="card-body">
                                        <!-- HS Code Section -->
                                        <div class="row mb-3">
                                            <div class="col-12">
                                                <label for="hs_code" class="form-label fw-semibold">HS Code *</label>
                                                <div class="input-group">
                                                    <input type="text" class="form-control" 
                                                           id="hs_code" name="hs_code" 
                                                           placeholder="e.g., 870321" required
                                                           value="{{ old('hs_code') }}">
                                                    <button type="button" class="btn btn-outline-primary" 
                                                            data-bs-toggle="modal" data-bs-target="#hsCodeModal">
                                                        <i class="fas fa-search me-1"></i>Browse
                                                    </button>
                                                </div>
                                                <div class="form-text">Enter 6-10 digit Harmonized System code</div>
                                                <div id="hsCodeDetails" class="mt-2"></div>
                                            </div>
                                        </div>

                                        <!-- Description -->
                                        <div class="row mb-3">
                                            <div class="col-12">
                                                <label for="description" class="form-label fw-semibold">Goods Description</label>
                                                <textarea class="form-control" id="description" name="description" 
                                                          rows="2" placeholder="Describe the goods being imported...">{{ old('description') }}</textarea>
                                            </div>
                                        </div>

                                        <!-- Country & Currency -->
                                        <div class="row mb-3">
                                            <div class="col-md-6">
                                                <label for="country_of_origin" class="form-label fw-semibold">Country of Origin *</label>
                                                <select class="form-select" id="country_of_origin" name="country_of_origin" required>
                                                    <option value="">Select Country</option>
                                                    @foreach($countries as $country)
                                                        <option value="{{ $country->code }}" {{ old('country_of_origin') == $country->code ? 'selected' : '' }}>
                                                            {{ $country->name }}
                                                        </option>
                                                    @endforeach
                                                </select>
                                            </div>
                                            <div class="col-md-6">
                                                <label for="currency" class="form-label fw-semibold">Transaction Currency</label>
                                                <select class="form-select" id="currency" name="currency">
                                                    @foreach($currencies as $currency)
                                                        <option value="{{ $currency->code }}" {{ old('currency', 'KES') == $currency->code ? 'selected' : '' }}>
                                                            {{ $currency->code }} - {{ $currency->name }}
                                                        </option>
                                                    @endforeach
                                                </select>
                                            </div>
                                        </div>

                                        <!-- Value & Quantity -->
                                        <div class="row mb-3">
                                            <div class="col-md-4">
                                                <label for="cif_value" class="form-label fw-semibold">CIF Value *</label>
                                                <input type="number" step="0.01" class="form-control" 
                                                       id="cif_value" name="cif_value" 
                                                       placeholder="0.00" required
                                                       value="{{ old('cif_value') }}">
                                                <div class="form-text">Cost + Insurance + Freight</div>
                                            </div>
                                            <div class="col-md-4">
                                                <label for="weight" class="form-label fw-semibold">Weight (KG)</label>
                                                <input type="number" step="0.01" class="form-control" 
                                                       id="weight" name="weight" 
                                                       placeholder="0.00"
                                                       value="{{ old('weight') }}">
                                            </div>
                                            <div class="col-md-4">
                                                <label for="quantity" class="form-label fw-semibold">Quantity</label>
                                                <input type="number" class="form-control" 
                                                       id="quantity" name="quantity" 
                                                       placeholder="1"
                                                       value="{{ old('quantity', 1) }}">
                                            </div>
                                        </div>

                                        <!-- Additional Options -->
                                        <div class="row mb-3">
                                            <div class="col-12">
                                                <div class="card bg-light border-0">
                                                    <div class="card-body py-3">
                                                        <h6 class="fw-semibold mb-3"><i class="fas fa-cog me-2"></i>Calculation Options</h6>
                                                        <div class="row">
                                                            <div class="col-md-4">
                                                                <div class="form-check form-switch">
                                                                    <input class="form-check-input" type="checkbox" id="include_vat" name="include_vat" value="1" checked>
                                                                    <label class="form-check-label" for="include_vat">
                                                                        Include VAT
                                                                    </label>
                                                                </div>
                                                            </div>
                                                            <div class="col-md-4">
                                                                <div class="form-check form-switch">
                                                                    <input class="form-check-input" type="checkbox" id="include_idf" name="include_idf" value="1" checked>
                                                                    <label class="form-check-label" for="include_idf">
                                                                        Include IDF
                                                                    </label>
                                                                </div>
                                                            </div>
                                                            <div class="col-md-4">
                                                                <div class="form-check form-switch">
                                                                    <input class="form-check-input" type="checkbox" id="include_rdl" name="include_rdl" value="1" checked>
                                                                    <label class="form-check-label" for="include_rdl">
                                                                        Include RDL
                                                                    </label>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div class="row mt-2">
                                                            <div class="col-md-6">
                                                                <div class="form-check form-switch">
                                                                    <input class="form-check-input" type="checkbox" id="force_benchmark" name="force_benchmark" value="1">
                                                                    <label class="form-check-label" for="force_benchmark">
                                                                        Force Benchmark
                                                                    </label>
                                                                </div>
                                                            </div>
                                                            <div class="col-md-6">
                                                                <div class="form-check form-switch">
                                                                    <input class="form-check-input" type="checkbox" id="log_calculation" name="log_calculation" value="1" checked>
                                                                    <label class="form-check-label" for="log_calculation">
                                                                        Log Calculation
                                                                    </label>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        <!-- Action Buttons -->
                                        <div class="row">
                                            <div class="col-12">
                                                <div class="d-grid gap-2">
                                                    <button type="submit" class="btn btn-primary btn-lg" id="calculateBtn">
                                                        <i class="fas fa-calculator me-2"></i>Calculate Taxes
                                                    </button>
                                                    <button type="button" class="btn btn-outline-secondary" id="resetForm">
                                                        <i class="fas fa-redo me-2"></i>Reset Form
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Right Column - Quick Preview & Information -->
                            <div class="col-md-6">
                                <!-- Quick Preview Card -->
                                <div class="card border-warning mb-4">
                                    <div class="card-header bg-warning text-dark">
                                        <h5 class="mb-0"><i class="fas fa-eye me-2"></i>Quick Preview</h5>
                                    </div>
                                    <div class="card-body">
                                        <div id="quickPreview" class="text-center text-muted">
                                            <i class="fas fa-calculator fa-3x mb-3"></i>
                                            <p>Enter HS code and values to see preview</p>
                                        </div>
                                        <div id="previewContent" style="display: none;">
                                            <div class="row text-center mb-3">
                                                <div class="col-6">
                                                    <small class="text-muted">HS Code</small>
                                                    <div id="previewHsCode" class="fw-bold"></div>
                                                </div>
                                                <div class="col-6">
                                                    <small class="text-muted">Duty Rate</small>
                                                    <div id="previewDuty" class="fw-bold"></div>
                                                </div>
                                            </div>
                                            <div class="progress mb-3">
                                                <div class="progress-bar progress-bar-striped progress-bar-animated" 
                                                     id="taxProgress" style="width: 0%"></div>
                                            </div>
                                            <div class="row text-center">
                                                <div class="col-4">
                                                    <small class="text-muted">Import Duty</small>
                                                    <div id="previewImportDuty" class="fw-bold text-primary">KES 0.00</div>
                                                </div>
                                                <div class="col-4">
                                                    <small class="text-muted">VAT</small>
                                                    <div id="previewVat" class="fw-bold text-success">KES 0.00</div>
                                                </div>
                                                <div class="col-4">
                                                    <small class="text-muted">Total</small>
                                                    <div id="previewTotal" class="fw-bold text-danger">KES 0.00</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- System Information Card -->
                                <div class="card border-info">
                                    <div class="card-header bg-info text-white">
                                        <h5 class="mb-0"><i class="fas fa-info-circle me-2"></i>System Information</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-6">
                                                <small class="text-muted">Active Tax Rules</small>
                                                <div class="fw-bold">{{ $activeTaxConfigs ?? 0 }} rules</div>
                                            </div>
                                            <div class="col-6">
                                                <small class="text-muted">HS Codes</small>
                                                <div class="fw-bold">{{ $totalHsCodes ?? 0 }} codes</div>
                                            </div>
                                        </div>
                                        <div class="row mt-2">
                                            <div class="col-6">
                                                <small class="text-muted">Today's Calculations</small>
                                                <div class="fw-bold">{{ $todayCalculations ?? 0 }}</div>
                                            </div>
                                            <div class="col-6">
                                                <small class="text-muted">Exchange Rates</small>
                                                <div class="fw-bold">{{ $activeExchangeRates ?? 0 }} active</div>
                                            </div>
                                        </div>
                                        <hr>
                                        <div class="text-center">
                                            <small class="text-muted">Last System Update</small>
                                            <div class="fw-bold">{{ isset($lastUpdate) ? $lastUpdate->format('M j, Y H:i') : now()->format('M j, Y H:i') }}</div>
                                        </div>
                                    </div>
                                                                    </div>

                                <!-- Quick Actions Card -->
                                <div class="card border-success mt-4">
                                    <div class="card-header bg-success text-white">
                                        <h5 class="mb-0"><i class="fas fa-bolt me-2"></i>Quick Actions</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="d-grid gap-2">
                                            <a href="{{ route('admin.hs-codes.create') }}" class="btn btn-outline-success btn-sm">
                                                <i class="fas fa-plus me-1"></i>Add HS Code
                                            </a>
                                            <a href="{{ route('admin.tax-configs.create') }}" class="btn btn-outline-primary btn-sm">
                                                <i class="fas fa-cog me-1"></i>Tax Configuration
                                            </a>
                                            <a href="{{ route('admin.exchange-rates.create') }}" class="btn btn-outline-warning btn-sm">
                                                <i class="fas fa-exchange-alt me-1"></i>Exchange Rates
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- HS Code Modal -->
@include('admin.tax-calculator.partials.hs-code-modal')

<!-- Quick Guide Modal -->
<div class="modal fade" id="quickGuideModal" tabindex="-1" aria-labelledby="quickGuideModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="quickGuideModalLabel">Tax Calculator Guide</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6><i class="fas fa-info-circle text-primary me-2"></i>Required Fields</h6>
                        <ul class="small">
                            <li><strong>HS Code:</strong> 6-10 digit code for product classification</li>
                            <li><strong>Country of Origin:</strong> Determines preferential duty rates</li>
                            <li><strong>CIF Value:</strong> Cost, Insurance, and Freight value</li>
                        </ul>
                    </div>
                    <div class="col-md-6">
                        <h6><i class="fas fa-cog text-warning me-2"></i>Options</h6>
                        <ul class="small">
                            <li><strong>Force Benchmark:</strong> Always use benchmark value</li>
                            <li><strong>Log Calculation:</strong> Save to calculation history</li>
                            <li><strong>Tax Inclusions:</strong> Toggle specific taxes on/off</li>
                        </ul>
                    </div>
                </div>
                <hr>
                <h6><i class="fas fa-calculator text-success me-2"></i>Calculation Steps</h6>
                <ol class="small">
                    <li>Validate HS code and fetch duty rates</li>
                    <li>Convert currency to KES if needed</li>
                    <li>Apply benchmark if CIF value is lower</li>
                    <li>Calculate import duty based on type</li>
                    <li>Compute VAT, IDF, RDL, and excise duties</li>
                    <li>Sum all applicable taxes</li>
                </ol>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
.card {
    border-radius: 10px;
}

.form-control, .form-select {
    border-radius: 6px;
}

.btn {
    border-radius: 6px;
}

.progress {
    border-radius: 10px;
    height: 8px;
}

.form-check-input:checked {
    background-color: #0d6efd;
    border-color: #0d6efd;
}
</style>
@endpush

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('adminTaxCalculatorForm');
    const calculateBtn = document.getElementById('calculateBtn');
    const resetBtn = document.getElementById('resetForm');
    const quickPreview = document.getElementById('quickPreview');
    const previewContent = document.getElementById('previewContent');

    // Real-time HS code lookup
    document.getElementById('hs_code').addEventListener('blur', function() {
        const code = this.value.trim();
        if (code.length >= 4) {
            showHsCodeDetails(code);
            updateQuickPreview();
        }
    });

    // Real-time preview updates
    document.getElementById('cif_value').addEventListener('input', updateQuickPreview);
    document.getElementById('weight').addEventListener('input', updateQuickPreview);
    document.getElementById('quantity').addEventListener('input', updateQuickPreview);
    document.getElementById('currency').addEventListener('change', updateQuickPreview);

    // Reset form
    resetBtn.addEventListener('click', function() {
        form.reset();
        quickPreview.style.display = 'block';
        previewContent.style.display = 'none';
        document.getElementById('hsCodeDetails').innerHTML = '';
    });

    // Form submission
    form.addEventListener('submit', function(e) {
        calculateBtn.disabled = true;
        calculateBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Calculating...';
    });

    function showHsCodeDetails(code) {
        fetch(`/api/hs-codes/${code}`)
            .then(response => response.json())
            .then(data => {
                const detailsDiv = document.getElementById('hsCodeDetails');
                if (data.error) {
                    detailsDiv.innerHTML = `<div class="alert alert-warning py-2">
                        <i class="fas fa-exclamation-triangle me-2"></i>${data.message || 'HS Code not found'}
                    </div>`;
                } else {
                    detailsDiv.innerHTML = `
                        <div class="alert alert-success py-2">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <strong>${data.hs_code}</strong>: ${data.description}
                                </div>
                                <div>
                                    <span class="badge bg-info me-2">${data.duty_type}</span>
                                    ${data.excise_applicable ? '<span class="badge bg-warning">Excise</span>' : ''}
                                    ${data.sensitive ? '<span class="badge bg-danger">Sensitive</span>' : ''}
                                </div>
                            </div>
                        </div>
                    `;
                }
            })
            .catch(error => {
                console.error('Error:', error);
            });
    }

    function updateQuickPreview() {
        const hsCode = document.getElementById('hs_code').value.trim();
        const cifValue = document.getElementById('cif_value').value;
        
        if (hsCode.length >= 4 && cifValue > 0) {
            // Simulate quick calculation
            fetch('/admin/tax-calculator/quick-preview', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    hs_code: hsCode,
                    cif_value: cifValue,
                    currency: document.getElementById('currency').value,
                    weight: document.getElementById('weight').value || 0,
                    quantity: document.getElementById('quantity').value || 1
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    quickPreview.style.display = 'none';
                    previewContent.style.display = 'block';
                    
                    document.getElementById('previewHsCode').textContent = data.hs_code;
                    document.getElementById('previewDuty').textContent = 
                        data.duty_type === 'percentage' ? data.duty_value + '%' : 'Flat';
                    
                    document.getElementById('previewImportDuty').textContent = 
                        'KES ' + data.import_duty.toLocaleString('en-KE', {minimumFractionDigits: 2});
                    document.getElementById('previewVat').textContent = 
                        'KES ' + data.vat.toLocaleString('en-KE', {minimumFractionDigits: 2});
                    document.getElementById('previewTotal').textContent = 
                        'KES ' + data.total_taxes.toLocaleString('en-KE', {minimumFractionDigits: 2});
                    
                    // Update progress bar
                    const progress = (data.import_duty / data.total_taxes) * 100;
                    document.getElementById('taxProgress').style.width = Math.min(progress, 100) + '%';
                }
            })
            .catch(error => {
                console.error('Preview error:', error);
            });
        }
    }
});
</script>
@endpush