@extends('layouts.admin')

@section('title', 'Add Exchange Rate')

@section('content')
<div class="container-fluid px-4">
    <div class="row">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">
                            <i class="fas fa-plus me-2"></i>Add New Exchange Rate
                        </h4>
                        <a href="{{ route('admin.exchange-rates.index') }}" class="btn btn-light btn-sm">
                            <i class="fas fa-arrow-left me-1"></i>Back to Rates
                        </a>
                    </div>
                </div>
                <div class="card-body">
                    <form action="{{ route('admin.exchange-rates.store') }}" method="POST">
                        @csrf
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="base_currency" class="form-label fw-semibold">Base Currency *</label>
                                    <select class="form-control @error('base_currency') is-invalid @enderror" 
                                            id="base_currency" name="base_currency" required>
                                        <option value="">Select Base Currency</option>
                                        @foreach($currencies as $currency)
                                            <option value="{{ $currency->code }}" 
                                                {{ old('base_currency', 'KES') == $currency->code ? 'selected' : '' }}
                                                {{ $currency->code == 'KES' ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    <div class="form-text">The base currency for the exchange rate</div>
                                    @error('base_currency')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="target_currency" class="form-label fw-semibold">Target Currency *</label>
                                    <select class="form-control @error('target_currency') is-invalid @enderror" 
                                            id="target_currency" name="target_currency" required>
                                        <option value="">Select Target Currency</option>
                                        @foreach($currencies as $currency)
                                            @if($currency->code != 'KES')
                                            <option value="{{ $currency->code }}" 
                                                {{ old('target_currency', request('target_currency')) == $currency->code ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                            @endif
                                        @endforeach
                                    </select>
                                    <div class="form-text">The currency being converted to</div>
                                    @error('target_currency')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="rate" class="form-label fw-semibold">Exchange Rate *</label>
                                    <input type="number" step="0.0001" class="form-control @error('rate') is-invalid @enderror" 
                                           id="rate" name="rate" value="{{ old('rate') }}" 
                                           placeholder="e.g., 150.5000" required>
                                    <div class="form-text">
                                        Rate format: 1 Target Currency = X Base Currency
                                        @if(old('base_currency', 'KES') && old('target_currency'))
                                            <br>1 {{ old('target_currency') }} = <span id="rateFormat">X</span> {{ old('base_currency', 'KES') }}
                                        @endif
                                    </div>
                                    @error('rate')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="effective_date" class="form-label fw-semibold">Effective Date *</label>
                                    <input type="date" class="form-control @error('effective_date') is-invalid @enderror" 
                                           id="effective_date" name="effective_date" 
                                           value="{{ old('effective_date', date('Y-m-d')) }}" required>
                                    <div class="form-text">The date when this rate becomes effective</div>
                                    @error('effective_date')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                               value="1" {{ old('is_active', true) ? 'checked' : '' }}>
                                        <label class="form-check-label fw-semibold" for="is_active">
                                            Active Rate
                                        </label>
                                    </div>
                                    <div class="form-text">Active rates are used in calculations</div>
                                </div>
                            </div>
                        </div>

                        <!-- Rate Preview -->
                        <div class="card bg-light mt-3" id="ratePreview" style="display: none;">
                            <div class="card-body">
                                <h6 class="text-primary mb-3">Rate Preview</h6>
                                <div class="row text-center">
                                    <div class="col-4">
                                        <small class="text-muted">Base Currency</small>
                                        <div id="previewBase" class="fw-bold"></div>
                                    </div>
                                    <div class="col-4">
                                        <small class="text-muted">Target Currency</small>
                                        <div id="previewTarget" class="fw-bold"></div>
                                    </div>
                                    <div class="col-4">
                                        <small class="text-muted">Exchange Rate</small>
                                        <div id="previewRate" class="fw-bold text-success"></div>
                                    </div>
                                </div>
                                <div class="text-center mt-2">
                                    <small class="text-muted" id="previewText"></small>
                                </div>
                            </div>
                        </div>

                        <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                            <a href="{{ route('admin.exchange-rates.index') }}" class="btn btn-secondary me-md-2">
                                <i class="fas fa-times me-2"></i>Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-2"></i>Create Exchange Rate
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-md-4">
            <!-- Help Information -->
            <div class="card border-info">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="fas fa-info-circle me-2"></i>Exchange Rate Guide</h5>
                </div>
                <div class="card-body">
                    <h6>Rate Format</h6>
                    <p class="small text-muted">
                        The exchange rate represents how much of the base currency equals one unit of the target currency.
                    </p>
                    <div class="bg-light p-3 rounded mb-3">
                        <strong>1 Target Currency = X Base Currency</strong>
                    </div>
                    
                    <h6>Examples</h6>
                    <ul class="small text-muted">
                        <li><strong>USD to KES:</strong> 1 USD = 150.50 KES</li>
                        <li><strong>EUR to KES:</strong> 1 EUR = 165.25 KES</li>
                        <li><strong>GBP to KES:</strong> 1 GBP = 190.75 KES</li>
                    </ul>
                    
                    <h6>Effective Date</h6>
                    <p class="small text-muted">
                        The system will use the most recent active rate on or before the calculation date.
                    </p>
                </div>
            </div>

            <!-- Recent Rates -->
            <div class="card border-warning mt-4">
                <div class="card-header bg-warning text-dark">
                    <h5 class="mb-0"><i class="fas fa-clock me-2"></i>Recent Rates</h5>
                </div>
                <div class="card-body">
                    @php
                        $recentRates = \App\Models\ExchangeRate::with(['baseCurrency', 'targetCurrency'])
                            ->latest('effective_date')
                            ->limit(5)
                            ->get();
                    @endphp
                    
                    @if($recentRates->count())
                        @foreach($recentRates as $recentRate)
                        <div class="border-bottom pb-2 mb-2">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <strong>{{ $recentRate->target_currency }}/{{ $recentRate->base_currency }}</strong>
                                    <br>
                                    <small class="text-muted">{{ number_format($recentRate->rate, 4) }}</small>
                                </div>
                                <div class="text-end">
                                    <small class="text-muted">{{ $recentRate->effective_date->format('M j') }}</small>
                                    <br>
                                    @if($recentRate->is_active)
                                        <span class="badge bg-success">Active</span>
                                    @else
                                        <span class="badge bg-secondary">Inactive</span>
                                    @endif
                                </div>
                            </div>
                        </div>
                        @endforeach
                    @else
                        <p class="text-muted small">No recent exchange rates found.</p>
                    @endif
                    
                    <div class="text-center mt-3">
                        <a href="{{ route('admin.exchange-rates.index') }}" class="btn btn-outline-warning btn-sm">
                            View All Rates
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const baseCurrencySelect = document.getElementById('base_currency');
    const targetCurrencySelect = document.getElementById('target_currency');
    const rateInput = document.getElementById('rate');
    const ratePreview = document.getElementById('ratePreview');
    const previewBase = document.getElementById('previewBase');
    const previewTarget = document.getElementById('previewTarget');
    const previewRate = document.getElementById('previewRate');
    const previewText = document.getElementById('previewText');

    function updateRatePreview() {
        const baseCurrency = baseCurrencySelect.value;
        const targetCurrency = targetCurrencySelect.value;
        const rate = rateInput.value;

        if (baseCurrency && targetCurrency && rate) {
            previewBase.textContent = baseCurrency;
            previewTarget.textContent = targetCurrency;
            previewRate.textContent = numberWithCommas(rate);
            previewText.textContent = `1 ${targetCurrency} = ${numberWithCommas(rate)} ${baseCurrency}`;
            ratePreview.style.display = 'block';
        } else {
            ratePreview.style.display = 'none';
        }
    }

    function numberWithCommas(x) {
        return parseFloat(x).toLocaleString('en-US', {
            minimumFractionDigits: 4,
            maximumFractionDigits: 4
        });
    }

    // Update preview when inputs change
    baseCurrencySelect.addEventListener('change', updateRatePreview);
    targetCurrencySelect.addEventListener('change', updateRatePreview);
    rateInput.addEventListener('input', updateRatePreview);

    // Auto-fill target currency from URL parameter
    const urlParams = new URLSearchParams(window.location.search);
    const targetCurrencyParam = urlParams.get('target_currency');
    if (targetCurrencyParam && targetCurrencySelect) {
        targetCurrencySelect.value = targetCurrencyParam;
        updateRatePreview();
    }

    // Initialize preview if values are already set
    updateRatePreview();
});
</script>
@endpush

@push('styles')
<style>
.card {
    border-radius: 10px;
}

.form-control, .form-select {
    border-radius: 6px;
}

.btn {
    border-radius: 6px;
}

.badge {
    font-size: 0.75em;
}
</style>
@endpush