@extends('layouts.admin')

@section('title', 'Calculation Logs')

@section('content')
<div class="container-fluid px-4">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">
                            <i class="fas fa-calculator me-2"></i>Calculation Logs
                        </h4>
                        <div class="btn-group">
                            <button type="button" class="btn btn-light btn-sm" data-bs-toggle="modal" data-bs-target="#exportModal">
                                <i class="fas fa-download me-1"></i>Export
                            </button>
                            <button type="button" class="btn btn-light btn-sm" data-bs-toggle="modal" data-bs-target="#bulkDeleteModal">
                                <i class="fas fa-trash me-1"></i>Bulk Delete
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Statistics -->
                    <div class="row mb-4">
                        <div class="col-md-2">
                            <div class="card border-primary text-center">
                                <div class="card-body py-3">
                                    <h3 class="text-primary mb-0">{{ number_format($stats['totalCalculations']) }}</h3>
                                    <small class="text-muted">Total</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <div class="card border-success text-center">
                                <div class="card-body py-3">
                                    <h3 class="text-success mb-0">{{ number_format($stats['todayCalculations']) }}</h3>
                                    <small class="text-muted">Today</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <div class="card border-info text-center">
                                <div class="card-body py-3">
                                    <h3 class="text-info mb-0">{{ number_format($stats['weekCalculations']) }}</h3>
                                    <small class="text-muted">This Week</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card border-warning text-center">
                                <div class="card-body py-3">
                                    <h3 class="text-warning mb-0">KES {{ number_format($stats['totalTaxAmount'], 2) }}</h3>
                                    <small class="text-muted">Total Taxes</small>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card border-danger text-center">
                                <div class="card-body py-3">
                                    <h3 class="text-danger mb-0">KES {{ number_format($stats['averageTaxAmount'], 2) }}</h3>
                                    <small class="text-muted">Average Tax</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Filters -->
                    <form action="{{ route('admin.calculation-logs.index') }}" method="GET" class="mb-4">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label for="search" class="form-label">Search</label>
                                    <input type="text" class="form-control" id="search" name="search" 
                                           value="{{ request('search') }}" placeholder="HS Code, Description, Country...">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group">
                                    <label for="user_id" class="form-label">User</label>
                                    <select class="form-control" id="user_id" name="user_id">
                                        <option value="">All Users</option>
                                        @foreach($users as $user)
                                            <option value="{{ $user->id }}" 
                                                {{ request('user_id') == $user->id ? 'selected' : '' }}>
                                                {{ $user->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group">
                                    <label for="min_amount" class="form-label">Min Amount (KES)</label>
                                    <input type="number" class="form-control" id="min_amount" name="min_amount" 
                                           value="{{ request('min_amount') }}" placeholder="0">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group">
                                    <label for="max_amount" class="form-label">Max Amount (KES)</label>
                                    <input type="number" class="form-control" id="max_amount" name="max_amount" 
                                           value="{{ request('max_amount') }}" placeholder="1000000">
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="row">
                                    <div class="col-6">
                                        <div class="form-group">
                                            <label for="date_from" class="form-label">From Date</label>
                                            <input type="date" class="form-control" id="date_from" name="date_from" 
                                                   value="{{ request('date_from') }}">
                                        </div>
                                    </div>
                                    <div class="col-6">
                                        <div class="form-group">
                                            <label for="date_to" class="form-label">To Date</label>
                                            <input type="date" class="form-control" id="date_to" name="date_to" 
                                                   value="{{ request('date_to') }}">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-12">
                                <button type="submit" class="btn btn-primary">Apply Filters</button>
                                <a href="{{ route('admin.calculation-logs.index') }}" class="btn btn-secondary">Clear Filters</a>
                            </div>
                        </div>
                    </form>

                    @if($calculationLogs->count())
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th width="5%">
                                        <input type="checkbox" id="selectAll" class="form-check-input">
                                    </th>
                                    <th width="10%">ID</th>
                                    <th width="15%">HS Code</th>
                                    <th width="20%">Description</th>
                                    <th width="10%">Country</th>
                                    <th width="15%">Tax Amount</th>
                                    <th width="15%">User</th>
                                    <th width="10%">Date</th>
                                    <th width="10%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($calculationLogs as $log)
                                <tr>
                                    <td>
                                        <input type="checkbox" name="selected_ids[]" value="{{ $log->id }}" class="form-check-input row-checkbox">
                                    </td>
                                    <td>
                                        <small class="text-muted">#{{ $log->id }}</small>
                                    </td>
                                    <td>
                                        <code>{{ $log->getInputValue('hs_code', 'N/A') }}</code>
                                    </td>
                                    <td>
                                        <div class="text-truncate" style="max-width: 200px;" title="{{ $log->getInputValue('description', 'No description') }}">
                                            {{ $log->getInputValue('description', 'No description') }}
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge bg-info">{{ $log->getInputValue('country_of_origin', 'N/A') }}</span>
                                    </td>
                                    <td>
                                        <strong class="text-success">KES {{ number_format($log->total_payable, 2) }}</strong>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="flex-shrink-0">
                                                <i class="fas fa-user-circle text-muted"></i>
                                            </div>
                                            <div class="flex-grow-1 ms-2">
                                                <div class="fw-semibold">{{ $log->user->name ?? 'System' }}</div>
                                                <small class="text-muted">{{ $log->user->email ?? 'N/A' }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $log->created_at->format('M j, Y') }}</small>
                                        <br>
                                        <small class="text-muted">{{ $log->created_at->format('H:i') }}</small>
                                    </td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <a href="{{ route('admin.calculation-logs.show', $log->id) }}" 
                                               class="btn btn-info" title="View Details">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <a href="{{ route('admin.calculation-logs.recalculate', $log->id) }}" 
                                               class="btn btn-warning" title="Recalculate"
                                               onclick="return confirm('Are you sure you want to recalculate this entry?')">
                                                <i class="fas fa-redo"></i>
                                            </a>
                                            <form action="{{ route('admin.calculation-logs.destroy', $log->id) }}" 
                                                  method="POST" class="d-inline">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn btn-danger" 
                                                        title="Delete"
                                                        onclick="return confirm('Are you sure you want to delete this calculation log?')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="d-flex justify-content-between align-items-center mt-3">
                        <div class="text-muted">
                            Showing {{ $calculationLogs->firstItem() }} to {{ $calculationLogs->lastItem() }} of {{ $calculationLogs->total() }} calculations
                        </div>
                        {{ $calculationLogs->links() }}
                    </div>
                    @else
                    <div class="text-center py-5">
                        <i class="fas fa-calculator fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">No Calculation Logs Found</h5>
                        <p class="text-muted mb-4">
                            @if(request()->anyFilled(['search', 'user_id', 'min_amount', 'max_amount', 'date_from', 'date_to']))
                                No calculation logs match your filter criteria.
                            @else
                                No tax calculations have been performed yet.
                            @endif
                        </p>
                        @if(request()->anyFilled(['search', 'user_id', 'min_amount', 'max_amount', 'date_from', 'date_to']))
                            <a href="{{ route('admin.calculation-logs.index') }}" class="btn btn-secondary">Clear Filters</a>
                        @else
                            <a href="{{ route('admin.tax-calculator') }}" class="btn btn-primary">
                                <i class="fas fa-calculator me-2"></i>Perform First Calculation
                            </a>
                        @endif
                    </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Export Modal -->
<div class="modal fade" id="exportModal" tabindex="-1" aria-labelledby="exportModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="exportModalLabel">
                    <i class="fas fa-download me-2"></i>Export Calculation Logs
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="list-group">
                    <a href="{{ route('admin.calculation-logs.export', 'today') }}" class="list-group-item list-group-item-action">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1">Today's Calculations</h6>
                            <small class="text-muted">{{ $stats['todayCalculations'] }} records</small>
                        </div>
                        <small class="text-muted">Export all calculations from today</small>
                    </a>
                    <a href="{{ route('admin.calculation-logs.export', 'week') }}" class="list-group-item list-group-item-action">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1">This Week's Calculations</h6>
                            <small class="text-muted">{{ $stats['weekCalculations'] }} records</small>
                        </div>
                        <small class="text-muted">Export all calculations from this week</small>
                    </a>
                    <a href="{{ route('admin.calculation-logs.export', 'month') }}" class="list-group-item list-group-item-action">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1">This Month's Calculations</h6>
                            <small class="text-muted">{{ $stats['monthCalculations'] }} records</small>
                        </div>
                        <small class="text-muted">Export all calculations from this month</small>
                    </a>
                    <a href="{{ route('admin.calculation-logs.export', 'all') }}" class="list-group-item list-group-item-action">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1">All Calculations</h6>
                            <small class="text-muted">{{ $stats['totalCalculations'] }} records</small>
                        </div>
                        <small class="text-muted">Export all calculation logs</small>
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Delete Modal -->
<div class="modal fade" id="bulkDeleteModal" tabindex="-1" aria-labelledby="bulkDeleteModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="bulkDeleteModalLabel">
                    <i class="fas fa-trash me-2"></i>Bulk Delete Calculation Logs
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('admin.calculation-logs.bulk-delete') }}" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <strong>Warning:</strong> This action cannot be undone. Selected calculation logs will be permanently deleted.
                    </div>
                    
                    <div class="form-group mb-3">
                        <label for="confirmation" class="form-label">Type "DELETE_CALCULATIONS" to confirm:</label>
                        <input type="text" class="form-control" id="confirmation" name="confirmation" 
                               placeholder="DELETE_CALCULATIONS" required>
                    </div>
                    
                    <input type="hidden" name="ids" id="selectedIds">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Selected</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Select all functionality
    const selectAll = document.getElementById('selectAll');
    const rowCheckboxes = document.querySelectorAll('.row-checkbox');
    
    selectAll.addEventListener('change', function() {
        rowCheckboxes.forEach(checkbox => {
            checkbox.checked = selectAll.checked;
        });
    });
    
    // Update select all when individual checkboxes change
    rowCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const allChecked = Array.from(rowCheckboxes).every(cb => cb.checked);
            selectAll.checked = allChecked;
        });
    });
    
    // Bulk delete modal
    const bulkDeleteModal = document.getElementById('bulkDeleteModal');
    if (bulkDeleteModal) {
        bulkDeleteModal.addEventListener('show.bs.modal', function() {
            const selectedIds = Array.from(rowCheckboxes)
                .filter(cb => cb.checked)
                .map(cb => cb.value);
            
            if (selectedIds.length === 0) {
                alert('Please select at least one calculation log to delete.');
                return false;
            }
            
            document.getElementById('selectedIds').value = JSON.stringify(selectedIds);
        });
    }
});
</script>
@endpush

@push('styles')
<style>
.card {
    border-radius: 10px;
}

.table th {
    border-top: none;
    font-weight: 600;
}

.btn {
    border-radius: 6px;
}

.badge {
    font-size: 0.75em;
}

.text-truncate {
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}
</style>
@endpush