<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    public function up(): void
    {
        // Check if admin_roles table exists but has wrong structure
        if (Schema::hasTable('admin_roles')) {
            $this->fixAdminRolesTable();
        } else {
            // Create the table if it doesn't exist
            Schema::create('admin_roles', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->string('slug')->unique();
                $table->json('permissions')->nullable();
                $table->text('description')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
            
            $this->insertDefaultRoles();
        }

        // Add role_id to users table if it doesn't exist
        $this->addRoleIdToUsers();
    }

    private function fixAdminRolesTable(): void
    {
        $columns = Schema::getColumnListing('admin_roles');
        
        // Add missing columns
        if (!in_array('slug', $columns)) {
            Schema::table('admin_roles', function (Blueprint $table) {
                $table->string('slug')->nullable()->after('name');
            });
            
            // Update existing records with default slugs
            DB::table('admin_roles')->whereNull('slug')->update([
                'slug' => DB::raw('LOWER(REPLACE(name, " ", "_"))')
            ]);
            
            // Now make slug unique and not null
            Schema::table('admin_roles', function (Blueprint $table) {
                $table->string('slug')->unique()->nullable(false)->change();
            });
        }
        
        if (!in_array('permissions', $columns)) {
            Schema::table('admin_roles', function (Blueprint $table) {
                $table->json('permissions')->nullable()->after('slug');
            });
        }
        
        if (!in_array('description', $columns)) {
            Schema::table('admin_roles', function (Blueprint $table) {
                $table->text('description')->nullable()->after('permissions');
            });
        }
        
        if (!in_array('is_active', $columns)) {
            Schema::table('admin_roles', function (Blueprint $table) {
                $table->boolean('is_active')->default(true)->after('description');
            });
        }

        // Insert default roles if table is empty
        if (DB::table('admin_roles')->count() === 0) {
            $this->insertDefaultRoles();
        }
    }

    private function insertDefaultRoles(): void
    {
        // Insert default roles only if they don't exist
        if (!DB::table('admin_roles')->where('slug', 'super_admin')->exists()) {
            DB::table('admin_roles')->insert([
                [
                    'name' => 'Super Administrator',
                    'slug' => 'super_admin',
                    'permissions' => json_encode(['*']),
                    'description' => 'Full system access with all permissions',
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now(),
                ],
                [
                    'name' => 'Content Manager',
                    'slug' => 'content_manager',
                    'permissions' => json_encode([
                        'countries.*',
                        'faqs.*',
                        'settings.view',
                        'contact-submissions.view',
                        'calculation-logs.view'
                    ]),
                    'description' => 'Can manage countries, FAQs, and basic content',
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now(),
                ],
                [
                    'name' => 'Reports Manager',
                    'slug' => 'reports_manager',
                    'permissions' => json_encode([
                        'reports.*',
                        'calculation-logs.view',
                        'contact-submissions.view',
                        'audit-logs.view'
                    ]),
                    'description' => 'Can view reports and analytics',
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]
            ]);
        }
    }

    private function addRoleIdToUsers(): void
    {
        if (!Schema::hasColumn('users', 'role_id')) {
            Schema::table('users', function (Blueprint $table) {
                $table->foreignId('role_id')->nullable()->after('is_active')
                      ->constrained('admin_roles')->onDelete('set null');
            });
        }
    }

    public function down(): void
    {
        // Don't drop tables in fix migration to avoid data loss
    }
};