<?php

namespace App\Services;

use App\Models\HsCode;
use App\Models\TaxConfig;
use App\Models\CountryDutyOverride;
use App\Models\Benchmark;
use App\Models\Currency;
use App\Models\CalculationLog;
use App\Models\ExchangeRate;
use Illuminate\Support\Facades\DB;

class TaxCalculatorService
{
    public function calculate(array $input): array
    {
        return DB::transaction(function () use ($input) {
            // Step 1: Validate and find HS Code
            $hsCode = HsCode::where('hs_code', $input['hs_code'])->firstOrFail();
            
            // Step 2: Extract and convert currency if needed
            $cifValue = $this->normalizeCurrency($input);
            $weight = isset($input['weight']) ? floatval($input['weight']) : 0;
            $quantity = isset($input['quantity']) ? intval($input['quantity']) : 1;
            $countryCode = $input['country_of_origin'];

            // Step 3: Calculate tax base (CIF or Benchmark)
            $taxBaseCalculation = $this->calculateTaxBase($hsCode, $cifValue, $weight, $quantity);
            $taxBase = $taxBaseCalculation['tax_base'];
            $benchmarkApplied = $taxBaseCalculation['benchmark_applied'];
            $benchmarkValue = $taxBaseCalculation['benchmark_value'];

            // Step 4: Calculate import duty
            $importDuty = $this->calculateImportDuty($hsCode, $taxBase, $weight, $quantity, $countryCode);
            
            // Step 5: Calculate other taxes
            $exciseDuty = $hsCode->excise_applicable ? 
                $this->calculateExciseDuty($hsCode, $taxBase) : 0;
                
            $idf = $this->calculateIDF($taxBase);
            $rdl = $this->calculateRDL($taxBase);
            
            // Step 6: Calculate VAT (on tax base + ALL duties)
            $vat = $this->calculateVAT($taxBase, $importDuty, $exciseDuty, $idf, $rdl);

            // Step 7: Apply tax inclusions from user preferences
            $inclusions = $this->applyTaxInclusions($input, [
                'import_duty' => $importDuty,
                'excise_duty' => $exciseDuty,
                'vat' => $vat,
                'idf' => $idf,
                'rdl' => $rdl,
            ]);

            // Step 8: Calculate total taxes
            $totalTaxes = array_sum(array_values($inclusions));

            // Step 9: Prepare result
            $result = [
                'breakdown' => [
                    'import_duty' => round($inclusions['import_duty'], 2),
                    'excise_duty' => round($inclusions['excise_duty'], 2),
                    'vat' => round($inclusions['vat'], 2),
                    'idf' => round($inclusions['idf'], 2),
                    'rdl' => round($inclusions['rdl'], 2),
                ],
                'total_taxes' => round($totalTaxes, 2),
                'tax_base' => [
                    'cif_value' => round($cifValue, 2),
                    'benchmark_applied' => $benchmarkApplied,
                    'benchmark_value' => $benchmarkValue,
                    'final_tax_base' => round($taxBase, 2),
                ],
                'hs_code_details' => [
                    'code' => $hsCode->hs_code,
                    'description' => $hsCode->description,
                    'duty_type' => $hsCode->default_duty_type,
                    'duty_value' => $hsCode->default_duty_value,
                    'excise_applicable' => $hsCode->excise_applicable,
                    'sensitive' => $hsCode->sensitive,
                ],
                'calculation_steps' => [
                    'tax_base_calculation' => $taxBaseCalculation,
                    'import_duty_calculation' => [
                        'type' => $this->getDutyType($hsCode, $countryCode),
                        'value' => $this->getDutyValue($hsCode, $countryCode),
                    ],
                    'tax_inclusions_applied' => [
                        'include_vat' => $input['include_vat'] ?? true,
                        'include_idf' => $input['include_idf'] ?? true,
                        'include_rdl' => $input['include_rdl'] ?? true,
                    ],
                ],
            ];

            // Step 10: Log calculation
            $this->logCalculation($input, $result);

            return $result;
        });
    }

    private function normalizeCurrency(array $input): float
    {
        $cifValue = floatval($input['cif_value']);
        $currency = $input['currency'] ?? 'KES';

        if ($currency !== 'KES') {
            return $this->convertToKES($cifValue, $currency);
        }

        return $cifValue;
    }

    private function convertToKES(float $amount, string $fromCurrency): float
    {
        if ($fromCurrency === 'KES') {
            return $amount;
        }

        $rate = ExchangeRate::where('base_currency', $fromCurrency)
            ->where('target_currency', 'KES')
            ->where('is_active', true)
            ->latest('effective_date')
            ->first();

        if (!$rate) {
            // Fallback to default rates if no exchange rate found
            $defaultRates = [
                'USD' => 150.00,
                'EUR' => 160.00,
                'GBP' => 190.00,
            ];

            if (isset($defaultRates[$fromCurrency])) {
                return $amount * $defaultRates[$fromCurrency];
            }

            throw new \Exception("No exchange rate found for {$fromCurrency} to KES");
        }

        return $amount * $rate->rate;
    }

    private function calculateTaxBase(HsCode $hsCode, float $cifValue, float $weight, int $quantity): array
    {
        // Get active benchmark for this HS code
        $benchmark = Benchmark::where('hs_code_id', $hsCode->id)
            ->active()
            ->first();

        if (!$benchmark) {
            return [
                'tax_base' => $cifValue,
                'benchmark_applied' => false,
                'benchmark_value' => null,
                'benchmark_details' => null,
            ];
        }

        // Calculate benchmark value based on type
        $benchmarkValue = match($benchmark->min_type) {
            'cif' => $benchmark->min_value,
            'per_kg' => $benchmark->min_value * $weight,
            'per_container' => $benchmark->min_value * $quantity,
            default => $benchmark->min_value,
        };

        // Convert to KES if needed
        if ($benchmark->currency !== 'KES') {
            $benchmarkValue = $this->convertToKES($benchmarkValue, $benchmark->currency);
        }

        // Use the higher of CIF or Benchmark
        if ($cifValue < $benchmarkValue) {
            return [
                'tax_base' => $benchmarkValue,
                'benchmark_applied' => true,
                'benchmark_value' => round($benchmarkValue, 2),
                'benchmark_details' => [
                    'min_type' => $benchmark->min_type,
                    'min_value' => $benchmark->min_value,
                    'currency' => $benchmark->currency,
                ],
            ];
        }

        return [
            'tax_base' => $cifValue,
            'benchmark_applied' => false,
            'benchmark_value' => round($benchmarkValue, 2),
            'benchmark_details' => [
                'min_type' => $benchmark->min_type,
                'min_value' => $benchmark->min_value,
                'currency' => $benchmark->currency,
            ],
        ];
    }

    private function calculateImportDuty(HsCode $hsCode, float $taxBase, float $weight, int $quantity, string $countryCode): float
    {
        // Check for country-specific override first
        $override = CountryDutyOverride::where('country_code', $countryCode)
            ->where('hs_code_id', $hsCode->id)
            ->active()
            ->first();

        if ($override) {
            return $this->calculateDutyByType($override->duty_type, $override->duty_value, $taxBase, $weight, $quantity);
        }

        // Use default HS code duty
        return $this->calculateDutyByType($hsCode->default_duty_type, $hsCode->default_duty_value, $taxBase, $weight, $quantity);
    }

    private function calculateDutyByType(string $type, float $value, float $taxBase, float $weight, int $quantity): float
    {
        return match($type) {
            'percentage' => ($taxBase * $value) / 100,
            'flat' => $value,
            'per_kg' => $weight * $value,
            'per_container' => $quantity * $value,
            default => 0,
        };
    }

    private function calculateVAT(float $taxBase, float $importDuty, float $exciseDuty, float $idf, float $rdl): float
    {
        $vatConfig = TaxConfig::where('tax_name', 'VAT')
            ->active()
            ->current()
            ->first();

        if (!$vatConfig) {
            // Default VAT rate if not configured
            return ($taxBase + $importDuty + $exciseDuty + $idf + $rdl) * 0.16;
        }

        // VAT base = Tax Base + Import Duty + Excise Duty + IDF + RDL
        $vatBase = $taxBase + $importDuty + $exciseDuty + $idf + $rdl;
        
        if ($vatConfig->rate_type === 'percentage') {
            return ($vatBase * $vatConfig->rate) / 100;
        }

        return $vatConfig->rate;
    }

    private function calculateIDF(float $taxBase): float
    {
        $idfConfig = TaxConfig::where('tax_name', 'IDF')
            ->active()
            ->current()
            ->first();

        if (!$idfConfig) {
            // Default IDF rate if not configured
            return $taxBase * 0.025;
        }

        if ($idfConfig->rate_type === 'percentage') {
            $idf = ($taxBase * $idfConfig->rate) / 100;
            
            // Apply minimum amount if specified
            if ($idfConfig->min_amount && $idf < $idfConfig->min_amount) {
                return $idfConfig->min_amount;
            }
            
            return $idf;
        }

        return $idfConfig->rate;
    }

    private function calculateRDL(float $taxBase): float
    {
        $rdlConfig = TaxConfig::where('tax_name', 'RDL')
            ->active()
            ->current()
            ->first();

        if (!$rdlConfig) {
            // Default RDL rate if not configured
            return $taxBase * 0.02;
        }

        if ($rdlConfig->rate_type === 'percentage') {
            return ($taxBase * $rdlConfig->rate) / 100;
        }

        return $rdlConfig->rate;
    }

    private function calculateExciseDuty(HsCode $hsCode, float $taxBase): float
    {
        if (!$hsCode->excise_applicable || !$hsCode->excise_formula) {
            return 0;
        }

        $formula = $hsCode->excise_formula;
        
        // Handle JSON string or array
        if (is_string($formula)) {
            $formula = json_decode($formula, true);
        }

        if (!$formula || !isset($formula['type'])) {
            return 0;
        }

        if ($formula['type'] === 'percentage') {
            return ($taxBase * $formula['value']) / 100;
        }

        return $formula['value'];
    }

    private function applyTaxInclusions(array $input, array $taxes): array
    {
        // Apply user preferences for tax inclusions
        if (isset($input['include_vat']) && !$input['include_vat']) {
            $taxes['vat'] = 0;
        }

        if (isset($input['include_idf']) && !$input['include_idf']) {
            $taxes['idf'] = 0;
        }

        if (isset($input['include_rdl']) && !$input['include_rdl']) {
            $taxes['rdl'] = 0;
        }

        return $taxes;
    }

    private function getDutyType(HsCode $hsCode, string $countryCode): string
    {
        $override = CountryDutyOverride::where('country_code', $countryCode)
            ->where('hs_code_id', $hsCode->id)
            ->active()
            ->first();

        return $override ? $override->duty_type : $hsCode->default_duty_type;
    }

    private function getDutyValue(HsCode $hsCode, string $countryCode): float
    {
        $override = CountryDutyOverride::where('country_code', $countryCode)
            ->where('hs_code_id', $hsCode->id)
            ->active()
            ->first();

        return $override ? $override->duty_value : $hsCode->default_duty_value;
    }

    private function logCalculation(array $input, array $result): void
    {
        CalculationLog::create([
            'input' => $input,
            'result' => $result,
            'total_payable' => $result['total_taxes'],
            'calculated_by' => auth()->id(),
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }

    /**
     * Validate calculation inputs before processing
     */
    public function validateInputs(array $input): array
    {
        $errors = [];

        // Check HS code exists
        if (!HsCode::where('hs_code', $input['hs_code'])->exists()) {
            $errors[] = 'HS Code not found';
        }

        // Check country exists
        if (!\App\Models\Country::where('code', $input['country_of_origin'])->exists()) {
            $errors[] = 'Country not found';
        }

        // Validate numeric values
        if ($input['cif_value'] <= 0) {
            $errors[] = 'CIF value must be greater than 0';
        }

        if (isset($input['weight']) && $input['weight'] < 0) {
            $errors[] = 'Weight cannot be negative';
        }

        if (isset($input['quantity']) && $input['quantity'] < 1) {
            $errors[] = 'Quantity must be at least 1';
        }

        return $errors;
    }
}