<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    protected $fillable = [
        'name',
        'email',
        'password',
        'is_admin',
        'is_active',
        'timezone',
        'role_id'
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'is_admin' => 'boolean',
        'is_active' => 'boolean',
    ];

    public function role()
    {
        return $this->belongsTo(AdminRole::class, 'role_id');
    }

    /**
     * Relationship with audit logs
     */
    public function auditLogs()
    {
        return $this->hasMany(AuditLog::class);
    }

    /**
     * Relationship with calculation logs
     */
    public function calculationLogs()
    {
        return $this->hasMany(CalculationLog::class, 'calculated_by');
    }

    /**
     * Check if this user is the first admin in the system
     * Cached for better performance
     */
    public function isFirstAdmin()
    {
        if (!$this->is_admin) {
            return false;
        }

        // Use caching to avoid repeated database queries
        $firstAdminId = cache()->remember("first_admin_id", 3600, function () {
            return self::where('is_admin', true)->orderBy('id')->value('id');
        });

        return $firstAdminId === $this->id;
    }

    /**
     * Check if user has permission with proper fallbacks
     */
    public function hasPermission($permission)
    {
        // First admin has all permissions
        if ($this->isFirstAdmin()) {
            return true;
        }

        // Check if user is active admin with role
        if (!$this->is_admin || !$this->is_active || !$this->role) {
            return false;
        }

        // Super admin role has all permissions
        if ($this->is_super_admin) {
            return true;
        }

        return $this->checkRolePermission($permission);
    }

    /**
     * Check role-based permissions with proper wildcard handling
     */
    protected function checkRolePermission($permission)
    {
        $permissions = $this->role->permissions ?? [];

        // Check for global wildcard
        if (in_array('*', $permissions)) {
            return true;
        }

        // Check exact permission match
        if (in_array($permission, $permissions)) {
            return true;
        }

        // Check wildcard permissions (e.g., users.* should match users.create, users.edit, etc.)
        foreach ($permissions as $perm) {
            if (str_ends_with($perm, '.*')) {
                $prefix = rtrim($perm, '.*');
                if (str_starts_with($permission, $prefix . '.')) {
                    return true;
                }
            }
        }

        return false;
    }

    // Add this method to check multiple permissions
    public function hasAnyPermission(array $permissions): bool
    {
        foreach ($permissions as $permission) {
            if ($this->hasPermission($permission)) {
                return true;
            }
        }
        return false;
    }

    public function hasAllPermissions(array $permissions): bool
    {
        foreach ($permissions as $permission) {
            if (!$this->hasPermission($permission)) {
                return false;
            }
        }
        return true;
    }

    public function getIsSuperAdminAttribute()
    {
        return $this->role && $this->role->slug === 'super_admin';
    }

    public function getIsContentManagerAttribute()
    {
        return $this->role && $this->role->slug === 'content_manager';
    }

    public function getIsReportsManagerAttribute()
    {
        return $this->role && $this->role->slug === 'reports_manager';
    }

    // Permission check methods for common actions
    public function canManageUsers()
    {
        return $this->hasPermission('users.*') || $this->hasPermission('users.create');
    }

    public function canManageCountries()
    {
        return $this->hasPermission('countries.*') || $this->hasPermission('countries.create');
    }

    public function canViewReports()
    {
        return $this->hasPermission('reports.*') || $this->hasPermission('reports.view');
    }

    public function canViewLogs()
    {
        return $this->hasPermission('audit-logs.*') || $this->hasPermission('audit-logs.view');
    }

    public function canManageSettings()
    {
        return $this->hasPermission('settings.*') || $this->hasPermission('settings.edit');
    }

    public function canManageRoles()
    {
        return $this->hasPermission('users.*') || $this->hasPermission('roles.create');
    }

    /**
     * Check if user can perform action (with first admin override)
     */
    public function can($ability, $arguments = [])
    {
        // First admin can do everything
        if ($this->isFirstAdmin()) {
            return true;
        }

        return $this->hasPermission($ability);
    }
}