<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class HsCode extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'hs_code',
        'description',
        'default_duty_type',
        'default_duty_value',
        'excise_applicable',
        'excise_formula',
        'sensitive',
    ];

    protected $casts = [
        'default_duty_value' => 'decimal:4',
        'excise_applicable' => 'boolean',
        'excise_formula' => 'array',
        'sensitive' => 'boolean',
    ];

    public function benchmarks()
    {
        return $this->hasMany(Benchmark::class);
    }

    public function countryOverrides()
    {
        return $this->hasMany(CountryDutyOverride::class);
    }

    public function activeBenchmark()
    {
        return $this->hasOne(Benchmark::class)
            ->where('effective_from', '<=', now())
            ->where(function($query) {
                $query->whereNull('effective_to')
                      ->orWhere('effective_to', '>=', now());
            })
            ->latest('effective_from');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('hs_code', 'LIKE', "%{$search}%")
              ->orWhere('description', 'LIKE', "%{$search}%");
        });
    }

    public function scopeWithDutyType($query, $type)
    {
        return $query->where('default_duty_type', $type);
    }

    public function scopeWithExcise($query, $applicable = true)
    {
        return $query->where('excise_applicable', $applicable);
    }

    public function getExciseFormulaAttribute($value)
    {
        return $value ? json_decode($value, true) : null;
    }

    public function setExciseFormulaAttribute($value)
    {
        $this->attributes['excise_formula'] = $value ? json_encode($value) : null;
    }
}