<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Currency extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'code',
        'name',
        'symbol',
        'active',
        'decimal_places',
    ];

    protected $casts = [
        'active' => 'boolean',
    ];

    public function exchangeRates()
    {
        return $this->hasMany(ExchangeRate::class, 'target_currency', 'code');
    }

    public function baseExchangeRates()
    {
        return $this->hasMany(ExchangeRate::class, 'base_currency', 'code');
    }

    // Add this relationship for the latest rate
    public function latestRate()
    {
        return $this->hasOne(ExchangeRate::class, 'target_currency', 'code')
            ->where('base_currency', 'KES')
            ->where('is_active', true)
            ->latest('effective_date');
    }

    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    public function getCurrentExchangeRateAttribute()
    {
        return $this->exchangeRates()
            ->where('base_currency', 'KES')
            ->where('effective_date', '<=', now())
            ->where('is_active', true)
            ->orderBy('effective_date', 'desc')
            ->first();
    }

    public static function getExchangeRate($fromCurrency, $toCurrency, $date = null)
    {
        $date = $date ?: now();
        
        if ($fromCurrency === $toCurrency) {
            return 1.0;
        }

        // Try direct rate
        $rate = ExchangeRate::where('base_currency', $fromCurrency)
            ->where('target_currency', $toCurrency)
            ->where('effective_date', '<=', $date)
            ->where('is_active', true)
            ->orderBy('effective_date', 'desc')
            ->first();

        if ($rate) {
            return $rate->rate;
        }

        // Try inverse rate
        $inverseRate = ExchangeRate::where('base_currency', $toCurrency)
            ->where('target_currency', $fromCurrency)
            ->where('effective_date', '<=', $date)
            ->where('is_active', true)
            ->orderBy('effective_date', 'desc')
            ->first();

        if ($inverseRate) {
            return 1 / $inverseRate->rate;
        }

        return null;
    }
}