<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class CountryDutyOverride extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'country_duty_overrides';

    protected $fillable = [
        'country_code',
        'hs_code_id',
        'duty_type',
        'duty_value',
        'remarks',
        'effective_from',
        'effective_to',
    ];

    protected $casts = [
        'duty_value' => 'decimal:4',
        'effective_from' => 'date',
        'effective_to' => 'date',
    ];

    public function hsCode()
    {
        return $this->belongsTo(HsCode::class);
    }

    public function country()
    {
        return $this->belongsTo(Country::class, 'country_code', 'code');
    }

    public function scopeActive($query)
    {
        return $query->where('effective_from', '<=', now())
            ->where(function($q) {
                $q->whereNull('effective_to')
                  ->orWhere('effective_to', '>=', now());
            });
    }

    public function scopeForCountry($query, $countryCode)
    {
        return $query->where('country_code', $countryCode);
    }

    public function scopeForHsCode($query, $hsCodeId)
    {
        return $query->where('hs_code_id', $hsCodeId);
    }

    public function scopeEffectiveOn($query, $date = null)
    {
        $date = $date ?: now();
        return $query->where('effective_from', '<=', $date)
            ->where(function($q) use ($date) {
                $q->whereNull('effective_to')
                  ->orWhere('effective_to', '>=', $date);
            });
    }

    public function getIsActiveAttribute(): bool
    {
        return $this->effective_from <= now() && 
               (!$this->effective_to || $this->effective_to >= now());
    }
}