<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ContactSubmission extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'subject',
        'message',
        'newsletter',
        'ip_address',
        'user_agent',
        'read_at',
        'status',
        'message_count',
        'last_message_at',
        'assigned_to',
        'admin_response',
        'responded_at',
    ];

    protected $casts = [
        'newsletter' => 'boolean',
        'read_at' => 'datetime',
        'last_message_at' => 'datetime',
        'responded_at' => 'datetime',
    ];

    // Relationships
    public function messages()
    {
        return $this->hasMany(ContactMessage::class);
    }

    public function assignedTo()
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    // Scopes
    public function scopeUnread($query)
    {
        return $query->whereNull('read_at');
    }

    public function scopeRead($query)
    {
        return $query->whereNotNull('read_at');
    }

    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeNew($query)
    {
        return $query->where('status', 'new');
    }

    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    public function scopeAssignedTo($query, $userId)
    {
        return $query->where('assigned_to', $userId);
    }

    // Methods
    public function markAsRead()
    {
        $this->update(['read_at' => now()]);
    }

    public function markAsUnread()
    {
        $this->update(['read_at' => null]);
    }

    public function updateStatus($status)
    {
        $this->update(['status' => $status]);
    }

    public function assignTo($userId)
    {
        $this->update(['assigned_to' => $userId]);
    }

    public function unassign()
    {
        $this->update(['assigned_to' => null]);
    }

    public function isRead()
    {
        return !is_null($this->read_at);
    }

    public function isUnread()
    {
        return is_null($this->read_at);
    }

    public function getStatusBadge()
    {
        $badges = [
            'new' => 'bg-danger',
            'in_progress' => 'bg-warning',
            'resolved' => 'bg-success',
            'closed' => 'bg-secondary',
        ];

        return $badges[$this->status] ?? 'bg-secondary';
    }

    public function addMessage($type, $message, $userId = null)
    {
        $newMessage = $this->messages()->create([
            'type' => $type,
            'message' => $message,
            'user_id' => $userId,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);

        // Update submission counters
        $this->update([
            'message_count' => $this->messages()->count(),
            'last_message_at' => now(),
            'read_at' => $type === 'outgoing' ? null : now(), // Mark as unread when admin sends message
        ]);

        return $newMessage;
    }
}