<?php

namespace App\Http\Controllers;

use App\Models\ContactSubmission;
use App\Models\Setting;
use App\Models\Faq;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\ContactConfirmation;

class ContactController extends Controller
{
    public function index()
    {
        $faqs = Faq::published()->ordered()->get();
        $settings = [
            'whatsapp_number' => Setting::get('whatsapp_number', '+254700000000'),
            'contact_email' => Setting::get('contact_email', 'support@taxsimulator.co.ke'),
        ];

        return view('guest.contact', compact('faqs', 'settings'));
    }

    public function submit(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|min:10',
            'newsletter' => 'boolean',
        ]);

        try {
            // Create contact submission
            $submission = ContactSubmission::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'subject' => $validated['subject'],
                'message' => $validated['message'],
                'newsletter' => $validated['newsletter'] ?? false,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            // Send confirmation email
            Mail::to($validated['email'])->send(new ContactConfirmation($submission));

            return response()->json([
                'success' => true,
                'message' => 'Your message has been sent successfully. We will get back to you soon.'
            ]);

        } catch (\Exception $e) {
            \Log::error('Contact form submission failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'There was an error sending your message. Please try again.'
            ], 500);
        }
    }
}