<?php

namespace App\Http\Controllers;

use App\Models\Country;
use App\Models\HsCode;
use App\Models\CalculationLog;
use App\Services\TaxCalculatorService;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class CalculatorController extends Controller
{
    protected $calculatorService;

    public function __construct(TaxCalculatorService $calculatorService)
    {
        $this->calculatorService = $calculatorService;
    }

    /**
     * Display the tax calculator form
     */
    public function index()
    {
        $countries = Country::active()->orderBy('name')->get();
        $popularHsCodes = HsCode::limit(20)->orderBy('hs_code')->get();
        
        return view('calculator.index', compact('countries', 'popularHsCodes'));
    }

    /**
     * Process tax calculation
     */
    public function calculate(Request $request)
    {
        try {
            $validated = $this->validateCalculationRequest($request);

            // Check if HS code exists
            $hsCode = HsCode::where('hs_code', $validated['hs_code'])->first();
            if (!$hsCode) {
                return $this->showHsCodeNotFoundCard($validated['hs_code'], $validated);
            }

            // Pre-validate inputs
            $validationErrors = $this->calculatorService->validateInputs($validated);
            if (!empty($validationErrors)) {
                throw ValidationException::withMessages([
                    'calculation' => $validationErrors
                ]);
            }

            // Perform calculation
            $result = $this->calculatorService->calculate($validated);
            
            // Get the latest calculation log entry
            $calculation = CalculationLog::latest()->first();

            return view('calculator.results', [
                'input' => $validated,
                'result' => $result,
                'calculation' => $calculation
            ]);

        } catch (ValidationException $e) {
            return redirect()->back()
                ->with('error', implode(', ', $e->errors()['calculation'] ?? ['Validation failed']))
                ->withInput();

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->showHsCodeNotFoundCard($request->hs_code, $request->all());

        } catch (\Exception $e) {
            \Log::error('Tax calculation failed: ' . $e->getMessage(), [
                'input' => $request->all(),
                'exception' => $e
            ]);

            return redirect()->back()
                ->with('error', 'Calculation failed: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display HS Code not found card
     */
    private function showHsCodeNotFoundCard(string $hsCode, array $input = [])
    {
        $countries = Country::active()->orderBy('name')->get();
        $popularHsCodes = HsCode::limit(20)->orderBy('hs_code')->get();
        
        return view('calculator.hs-code-not-found', [
            'hs_code' => $hsCode,
            'input' => $input,
            'countries' => $countries,
            'popularHsCodes' => $popularHsCodes
        ]);
    }

    /**
     * Display calculation results by ID
     */
    public function results($id)
    {
        try {
            $calculation = CalculationLog::findOrFail($id);
            
            return view('calculator.results', [
                'input' => $calculation->input,
                'result' => $calculation->result,
                'calculation' => $calculation,
            ]);

        } catch (\Exception $e) {
            return redirect()->route('calculator')
                ->with('error', 'Calculation result not found.');
        }
    }

    /**
     * Get HS code details via AJAX
     */
    public function getHsCodeDetails($code)
    {
        try {
            $hsCode = HsCode::where('hs_code', $code)->first();
            
            if (!$hsCode) {
                return response()->json([
                    'error' => 'HS Code not found',
                    'code' => $code,
                    'message' => 'We don\'t recognize this HS code. Please confirm and try again.'
                ], 404);
            }

            return response()->json([
                'hs_code' => $hsCode->hs_code,
                'description' => $hsCode->description,
                'duty_type' => $hsCode->default_duty_type,
                'duty_value' => $hsCode->default_duty_value,
                'excise_applicable' => (bool) $hsCode->excise_applicable,
                'sensitive' => (bool) $hsCode->sensitive,
                'excise_formula' => $hsCode->excise_formula,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch HS code details',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search HS codes via AJAX
     */
    public function searchHsCodes(Request $request)
    {
        try {
            $search = $request->get('search', '');
            
            if (strlen($search) < 2) {
                return response()->json([]);
            }

            $hsCodes = HsCode::where('hs_code', 'LIKE', "%{$search}%")
                ->orWhere('description', 'LIKE', "%{$search}%")
                ->limit(20)
                ->get()
                ->map(function ($hsCode) {
                    return [
                        'id' => $hsCode->id,
                        'hs_code' => $hsCode->hs_code,
                        'description' => $hsCode->description,
                        'duty_type' => $hsCode->default_duty_type,
                        'duty_value' => $hsCode->default_duty_value,
                        'excise_applicable' => (bool) $hsCode->excise_applicable,
                        'sensitive' => (bool) $hsCode->sensitive,
                    ];
                });

            return response()->json($hsCodes);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Search failed',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Validate calculation request
     */
    private function validateCalculationRequest(Request $request): array
    {
        return $request->validate([
            'hs_code' => 'required|string|max:20',
            'country_of_origin' => 'required|string|size:2',
            'cif_value' => 'required|numeric|min:0',
            'currency' => 'required|string|in:KES,USD,EUR,GBP',
            'weight' => 'nullable|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'description' => 'nullable|string|max:500',
            'include_vat' => 'sometimes|boolean',
            'include_idf' => 'sometimes|boolean',
            'include_rdl' => 'sometimes|boolean',
        ], [
            'hs_code.required' => 'HS Code is required',
            'country_of_origin.required' => 'Country of origin is required',
            'cif_value.required' => 'CIF value is required',
            'cif_value.numeric' => 'CIF value must be a number',
            'cif_value.min' => 'CIF value must be greater than 0',
            'currency.in' => 'Please select a valid currency',
            'weight.numeric' => 'Weight must be a number',
            'weight.min' => 'Weight cannot be negative',
            'quantity.integer' => 'Quantity must be a whole number',
            'quantity.min' => 'Quantity must be at least 1',
        ]);
    }

    /**
     * Get calculation history for authenticated users
     */
    public function history()
    {
        if (!auth()->check()) {
            return redirect()->route('login')
                ->with('info', 'Please login to view your calculation history.');
        }

        $calculations = CalculationLog::where('calculated_by', auth()->id())
            ->latest()
            ->paginate(10);

        return view('calculator.history', compact('calculations'));
    }

    /**
     * Quick calculation via AJAX (for real-time preview)
     */
    public function quickCalculate(Request $request)
    {
        try {
            $validated = $request->validate([
                'hs_code' => 'required|string|max:20',
                'country_of_origin' => 'required|string|size:2',
                'cif_value' => 'required|numeric|min:0',
                'currency' => 'required|string|in:KES,USD,EUR,GBP',
                'weight' => 'nullable|numeric|min:0',
            ]);

            // Quick validation without full calculation
            $hsCode = HsCode::where('hs_code', $validated['hs_code'])->first();
            if (!$hsCode) {
                return response()->json([
                    'error' => true,
                    'message' => 'We don\'t recognize this HS code. Please confirm and try again.',
                    'code' => $validated['hs_code']
                ], 404);
            }

            // Return basic info for preview
            return response()->json([
                'hs_code' => $hsCode->hs_code,
                'description' => $hsCode->description,
                'duty_type' => $hsCode->default_duty_type,
                'duty_value' => $hsCode->default_duty_value,
                'excise_applicable' => (bool) $hsCode->excise_applicable,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Quick calculation failed',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}