<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\BaseController;
use App\Models\User;
use App\Models\AdminRole;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends BaseController
{
    public function index()
    {
        $users = User::with('role')
            ->where('is_admin', true)
            ->orderBy('name')
            ->paginate($this->perPage);

        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        // Middleware handles permission checks
        $roles = AdminRole::where('is_active', true)->get();
        return view('admin.users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        // Middleware handles permission checks

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8|confirmed',
            'role_id' => 'required|exists:admin_roles,id',
            'timezone' => 'required|string|timezone',
            'is_active' => 'boolean'
        ]);

        $validated['password'] = Hash::make($validated['password']);
        $validated['is_admin'] = true;
        $validated['email_verified_at'] = now();
        $validated['is_active'] = $request->has('is_active');

        User::create($validated);

        return $this->successResponse('User created successfully.', 'admin.users.index');
    }

    public function edit(User $user)
    {
        // Middleware handles permission checks
        $roles = AdminRole::where('is_active', true)->get();
        return view('admin.users.edit', compact('user', 'roles'));
    }

    public function update(Request $request, User $user)
    {
        // Middleware handles permission checks

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                Rule::unique('users')->ignore($user->id)
            ],
            'role_id' => 'required|exists:admin_roles,id',
            'timezone' => 'required|string|timezone',
            'is_active' => 'boolean'
        ]);

        // Protect first admin
        if ($user->isFirstAdmin()) {
            $validated['is_active'] = true;
            // Optional: Prevent role change for first admin
            // $validated['role_id'] = $user->role_id;
        }

        $user->update($validated);

        return $this->successResponse('User updated successfully.', 'admin.users.index');
    }

    public function destroy(User $user)
    {
        // Middleware handles permission checks

        // Prevent deleting the first admin
        if ($user->isFirstAdmin()) {
            return $this->errorResponse('Cannot delete the first administrator.');
        }

        // Prevent users from deleting themselves
        if ($user->id === auth()->id()) {
            return $this->errorResponse('You cannot delete your own account.');
        }

        $user->delete();

        return $this->successResponse('User deleted successfully.', 'admin.users.index');
    }

    public function toggleStatus(User $user)
    {
        // Middleware handles permission checks

        // Prevent deactivating the first admin
        if ($user->isFirstAdmin()) {
            return $this->errorResponse('Cannot deactivate the first administrator.');
        }

        // Prevent users from deactivating themselves
        if ($user->id === auth()->id()) {
            return $this->errorResponse('You cannot deactivate your own account.');
        }

        $user->update([
            'is_active' => !$user->is_active
        ]);

        $status = $user->is_active ? 'activated' : 'deactivated';
        return $this->successResponse("User {$status} successfully.");
    }

    public function resetPassword(Request $request, User $user)
    {
        // Middleware handles permission checks

        $validated = $request->validate([
            'password' => 'required|min:8|confirmed'
        ]);

        $user->update([
            'password' => Hash::make($validated['password'])
        ]);

        return $this->successResponse('Password reset successfully.');
    }
}