<?php

namespace App\Http\Controllers\Admin;

use App\Models\TaxConfig;
use Illuminate\Http\Request;

class TaxConfigController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        $query = TaxConfig::query();

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->where('tax_name', 'LIKE', "%{$request->search}%")
                  ->orWhere('remarks', 'LIKE', "%{$request->search}%");
            });
        }

        // Filters
        if ($request->has('tax_name') && $request->tax_name) {
            $query->where('tax_name', $request->tax_name);
        }

        if ($request->has('active') && $request->active !== '') {
            $query->where('active', $request->active);
        }

        $taxConfigs = $query->latest()->paginate($this->perPage);

        return view('admin.tax-configs.index', compact('taxConfigs'));
    }

    public function create()
    {
        return view('admin.tax-configs.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'tax_name' => 'required|in:VAT,IDF,RDL,Excise,Import Duty',
            'rate_type' => 'required|in:percentage,flat',
            'rate' => 'required|numeric|min:0',
            'min_amount' => 'nullable|numeric|min:0',
            'active' => 'boolean',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after:effective_from',
            'remarks' => 'nullable|string|max:500',
        ]);

        try {
            TaxConfig::create($validated);
            return redirect()->route('admin.tax-configs.index')
                ->with('success', 'Tax configuration created successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create tax configuration: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(TaxConfig $taxConfig)
    {
        return view('admin.tax-configs.show', compact('taxConfig'));
    }

    public function edit(TaxConfig $taxConfig)
    {
        return view('admin.tax-configs.edit', compact('taxConfig'));
    }

    public function update(Request $request, TaxConfig $taxConfig)
    {
        $validated = $request->validate([
            'tax_name' => 'required|in:VAT,IDF,RDL,Excise,Import Duty',
            'rate_type' => 'required|in:percentage,flat',
            'rate' => 'required|numeric|min:0',
            'min_amount' => 'nullable|numeric|min:0',
            'active' => 'boolean',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after:effective_from',
            'remarks' => 'nullable|string|max:500',
        ]);

        try {
            $taxConfig->update($validated);
            return redirect()->route('admin.tax-configs.index')
                ->with('success', 'Tax configuration updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update tax configuration: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(TaxConfig $taxConfig)
    {
        try {
            $taxConfig->delete();
            return redirect()->route('admin.tax-configs.index')
                ->with('success', 'Tax configuration deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete tax configuration: ' . $e->getMessage());
        }
    }
}