<?php

namespace App\Http\Controllers\Admin;

use App\Models\Country;
use App\Models\HsCode;
use App\Models\Currency;
use App\Models\TaxConfig;
use App\Models\ExchangeRate;
use App\Models\CalculationLog;
use App\Services\TaxCalculatorService;
use Illuminate\Http\Request;
use Carbon\Carbon;

class TaxCalculatorController extends BaseController
{
    protected $calculatorService;

    public function __construct(TaxCalculatorService $calculatorService)
    {
        $this->calculatorService = $calculatorService;
    }

    public function index()
    {
        try {
            $countries = Country::active()->orderBy('name')->get();
            $currencies = Currency::active()->orderBy('code')->get();
            $popularHsCodes = HsCode::limit(50)->orderBy('hs_code')->get();
            
            // Get statistics safely
            $activeTaxConfigs = TaxConfig::active()->count();
            $totalHsCodes = HsCode::count();
            $todayCalculations = CalculationLog::whereDate('created_at', today())->count();
            $activeExchangeRates = ExchangeRate::where('is_active', true)->count();
            
            // Get last update time safely
            $lastCalculation = CalculationLog::latest()->first();
            $lastUpdate = $lastCalculation ? $lastCalculation->created_at : now();

            return view('admin.tax-calculator.index', compact(
                'countries', 
                'currencies', 
                'popularHsCodes',
                'activeTaxConfigs',
                'totalHsCodes', 
                'todayCalculations',
                'activeExchangeRates',
                'lastUpdate'
            ));

        } catch (\Exception $e) {
            // Fallback if there are issues with the database
            return view('admin.tax-calculator.index', [
                'countries' => collect(),
                'currencies' => collect(),
                'popularHsCodes' => collect(),
                'activeTaxConfigs' => 0,
                'totalHsCodes' => 0,
                'todayCalculations' => 0,
                'activeExchangeRates' => 0,
                'lastUpdate' => now(),
            ])->with('error', 'Some system data is not available: ' . $e->getMessage());
        }
    }

    public function calculate(Request $request)
    {
        $validated = $request->validate([
            'hs_code' => 'required|string|max:20',
            'country_of_origin' => 'required|string|size:2',
            'cif_value' => 'required|numeric|min:0',
            'currency' => 'required|string|in:KES,USD,EUR,GBP',
            'weight' => 'nullable|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'description' => 'nullable|string|max:500',
            'include_vat' => 'sometimes|boolean',
            'include_idf' => 'sometimes|boolean',
            'include_rdl' => 'sometimes|boolean',
            'force_benchmark' => 'sometimes|boolean',
            'log_calculation' => 'sometimes|boolean',
        ]);

        try {
            // Check if HS code exists
            $hsCode = HsCode::where('hs_code', $validated['hs_code'])->first();
            if (!$hsCode) {
                return redirect()->back()
                    ->with('error', 'HS Code not found: ' . $validated['hs_code'])
                    ->withInput();
            }

            // Perform calculation
            $result = $this->calculatorService->calculate($validated);
            
            // Get calculation log if logging is enabled
            $calculation = null;
            if ($validated['log_calculation'] ?? true) {
                $calculation = CalculationLog::latest()->first();
            }

            return view('admin.tax-calculator.results', [
                'input' => $validated,
                'result' => $result,
                'calculation' => $calculation
            ]);

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Calculation failed: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function history()
    {
        $calculations = CalculationLog::with('user')
            ->latest()
            ->paginate(20);

        return view('admin.tax-calculator.history', compact('calculations'));
    }

    public function quickPreview(Request $request)
    {
        try {
            $validated = $request->validate([
                'hs_code' => 'required|string|max:20',
                'cif_value' => 'required|numeric|min:0',
                'currency' => 'required|string|in:KES,USD,EUR,GBP',
                'weight' => 'nullable|numeric|min:0',
                'quantity' => 'nullable|integer|min:1',
            ]);

            // Get HS code details
            $hsCode = HsCode::where('hs_code', $validated['hs_code'])->first();
            if (!$hsCode) {
                return response()->json([
                    'success' => false,
                    'message' => 'HS Code not found'
                ], 404);
            }

            // Simple preview calculation (simplified for performance)
            $cifValue = $validated['cif_value'];
            if ($validated['currency'] !== 'KES') {
                $rate = ExchangeRate::where('base_currency', $validated['currency'])
                    ->where('target_currency', 'KES')
                    ->where('is_active', true)
                    ->latest('effective_date')
                    ->first();
                
                if ($rate) {
                    $cifValue = $cifValue * $rate->rate;
                }
            }

            // Calculate import duty
            $importDuty = $this->calculateSimpleDuty(
                $hsCode->default_duty_type,
                $hsCode->default_duty_value,
                $cifValue,
                $validated['weight'] ?? 0,
                $validated['quantity'] ?? 1
            );

            // Calculate other taxes (simplified)
            $vat = ($cifValue + $importDuty) * 0.16; // Simplified VAT
            $idf = $cifValue * 0.025; // Simplified IDF
            $rdl = $cifValue * 0.02; // Simplified RDL

            $totalTaxes = $importDuty + $vat + $idf + $rdl;

            return response()->json([
                'success' => true,
                'hs_code' => $hsCode->hs_code,
                'duty_type' => $hsCode->default_duty_type,
                'duty_value' => $hsCode->default_duty_value,
                'import_duty' => round($importDuty, 2),
                'vat' => round($vat, 2),
                'idf' => round($idf, 2),
                'rdl' => round($rdl, 2),
                'total_taxes' => round($totalTaxes, 2),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Preview calculation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    private function calculateSimpleDuty($type, $value, $taxBase, $weight, $quantity)
    {
        return match($type) {
            'percentage' => ($taxBase * $value) / 100,
            'flat' => $value,
            'per_kg' => $weight * $value,
            'per_container' => $quantity * $value,
            default => 0,
        };
    }

    public function searchHsCodes(Request $request)
    {
        try {
            $search = $request->get('search', '');
            $dutyType = $request->get('duty_type', '');
            
            $query = HsCode::query();

            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('hs_code', 'LIKE', "%{$search}%")
                      ->orWhere('description', 'LIKE', "%{$search}%");
                });
            }

            if ($dutyType) {
                $query->where('default_duty_type', $dutyType);
            }

            $hsCodes = $query->limit(50)->get();

            return response()->json($hsCodes);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Search failed',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}