<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Response;

class SystemHealthController extends BaseController
{
    public function index()
    {
        $systemInfo = $this->getSystemInfo();
        $databaseInfo = $this->getDatabaseInfo();
        $storageInfo = $this->getStorageInfo();
        $cacheInfo = $this->getCacheInfo();
        $applicationInfo = $this->getApplicationInfo();
        $backupInfo = $this->getBackupInfo();

        return view('admin.system-health.index', compact(
            'systemInfo',
            'databaseInfo',
            'storageInfo',
            'cacheInfo',
            'applicationInfo',
            'backupInfo'
        ));
    }

    public function clearCache()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');

            return redirect()->route('admin.system-health.index')
                ->with('success', 'Cache cleared successfully!');
        } catch (\Exception $e) {
            return redirect()->route('admin.system-health.index')
                ->with('error', 'Failed to clear cache: ' . $e->getMessage());
        }
    }

    public function runBackup()
    {
        try {
            Artisan::call('backup:run');
            
            return redirect()->route('admin.system-health.index')
                ->with('success', 'Backup completed successfully!');
        } catch (\Exception $e) {
            return redirect()->route('admin.system-health.index')
                ->with('error', 'Backup failed: ' . $e->getMessage());
        }
    }

    public function downloadBackup($filename)
    {
        try {
            $filePath = storage_path('app/backups/' . $filename);
            
            if (!File::exists($filePath)) {
                return redirect()->route('admin.system-health.index')
                    ->with('error', 'Backup file not found.');
            }

            return Response::download($filePath, $filename);
        } catch (\Exception $e) {
            return redirect()->route('admin.system-health.index')
                ->with('error', 'Download failed: ' . $e->getMessage());
        }
    }

    public function deleteBackup($filename)
    {
        try {
            $filePath = storage_path('app/backups/' . $filename);
            
            if (!File::exists($filePath)) {
                return redirect()->route('admin.system-health.index')
                    ->with('error', 'Backup file not found.');
            }

            File::delete($filePath);

            return redirect()->route('admin.system-health.index')
                ->with('success', 'Backup deleted successfully!');
        } catch (\Exception $e) {
            return redirect()->route('admin.system-health.index')
                ->with('error', 'Delete failed: ' . $e->getMessage());
        }
    }

    private function getSystemInfo()
    {
        return [
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'N/A',
            'server_os' => php_uname('s') . ' ' . php_uname('r'),
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'timezone' => config('app.timezone'),
            'environment' => app()->environment(),
        ];
    }

    private function getDatabaseInfo()
    {
        try {
            $connection = config('database.default');
            $driver = config("database.connections.{$connection}.driver");
            $database = config("database.connections.{$connection}.database");
            
            $tables = DB::select('SHOW TABLES');
            $tableCount = count($tables);
            
            $connectionStatus = 'Connected';
            
            // Get some table sizes (for MySQL)
            if ($driver === 'mysql') {
                $tableSizes = DB::select("
                    SELECT 
                        TABLE_NAME as `table`,
                        ROUND((DATA_LENGTH + INDEX_LENGTH) / 1024 / 1024, 2) as `size_mb`,
                        TABLE_ROWS as `rows`
                    FROM information_schema.TABLES 
                    WHERE TABLE_SCHEMA = ?
                    ORDER BY (DATA_LENGTH + INDEX_LENGTH) DESC
                ", [$database]);
            } else {
                $tableSizes = [];
            }

            return [
                'driver' => $driver,
                'database' => $database,
                'connection_status' => $connectionStatus,
                'table_count' => $tableCount,
                'table_sizes' => $tableSizes,
            ];
        } catch (\Exception $e) {
            return [
                'driver' => 'Unknown',
                'database' => 'Unknown',
                'connection_status' => 'Disconnected',
                'table_count' => 0,
                'table_sizes' => [],
                'error' => $e->getMessage(),
            ];
        }
    }

    private function getStorageInfo()
    {
        $disk = Storage::disk('local');
        $publicDisk = Storage::disk('public');
        
        return [
            'local_disk_status' => $disk->exists('.') ? 'OK' : 'Error',
            'public_disk_status' => $publicDisk->exists('.') ? 'OK' : 'Error',
            'local_disk_free_space' => $this->formatBytes(disk_free_space(storage_path())),
            'public_disk_free_space' => $this->formatBytes(disk_free_space(public_path())),
        ];
    }

    private function getCacheInfo()
    {
        $driver = config('cache.default');
        
        try {
            Cache::put('health_check', 'ok', 10);
            $status = Cache::get('health_check') === 'ok' ? 'OK' : 'Error';
        } catch (\Exception $e) {
            $status = 'Error';
        }

        return [
            'driver' => $driver,
            'status' => $status,
            'prefix' => config('cache.prefix'),
        ];
    }

    private function getApplicationInfo()
    {
        return [
            'name' => config('app.name'),
            'url' => config('app.url'),
            'debug' => config('app.debug') ? 'Enabled' : 'Disabled',
            'maintenance_mode' => app()->isDownForMaintenance() ? 'Enabled' : 'Disabled',
            'last_calculation' => \App\Models\CalculationLog::latest()->first()?->created_at?->diffForHumans() ?? 'Never',
            'total_calculations' => \App\Models\CalculationLog::count(),
            'total_users' => \App\Models\User::count(),
        ];
    }

    private function getBackupInfo()
    {
        $backupPath = storage_path('app/backups');
        $backups = [];
        
        try {
            if (File::exists($backupPath)) {
                $files = File::files($backupPath);
                
                foreach ($files as $file) {
                    if (in_array($file->getExtension(), ['zip', 'sql'])) {
                        $backups[] = [
                            'name' => $file->getFilename(),
                            'size' => $this->formatBytes($file->getSize()),
                            'modified' => $file->getMTime(),
                            'path' => $file->getPathname(),
                        ];
                    }
                }
                
                // Sort by modification time (newest first)
                usort($backups, function($a, $b) {
                    return $b['modified'] <=> $a['modified'];
                });
            }
        } catch (\Exception $e) {
            // Backup directory might not exist yet
        }

        return [
            'backup_count' => count($backups),
            'backups' => $backups,
            'latest_backup' => $backups[0] ?? null,
            'total_backup_size' => $this->formatBytes(array_sum(array_map(function($backup) {
                return File::size($backup['path']);
            }, $backups))),
        ];
    }

    private function formatBytes($bytes, $precision = 2)
    {
        if ($bytes <= 0) return '0 B';
        
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        $bytes /= pow(1024, $pow);

        return round($bytes, $precision) . ' ' . $units[$pow];
    }
}