<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Country;
use App\Models\Currency;
use App\Models\TaxConfig;
use App\Models\HsCode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class SetupController extends Controller
{
    public function showSetupForm()
    {
        // Check if any admin exists - redirect to login if true
        if (User::where('is_admin', true)->exists()) {
            return redirect()->route('login');
        }

        return view('admin.setup');
    }

    public function setup(Request $request)
    {
        // Check if any admin exists - redirect to login if true
        if (User::where('is_admin', true)->exists()) {
            return redirect()->route('login');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8|confirmed',
        ]);

        try {
            DB::transaction(function () use ($request) {
                // Create admin user
                $admin = User::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'password' => Hash::make($request->password),
                    'is_admin' => true,
                    'is_active' => true,
                    'email_verified_at' => now(),
                ]);

                // Seed essential data
                $this->seedEssentialData();
            });

            // Auto-login the admin
            auth()->attempt([
                'email' => $request->email,
                'password' => $request->password
            ]);

            return redirect()->route('admin.dashboard')
                ->with('success', 'Admin account created successfully! System is now ready.');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Setup failed: ' . $e->getMessage()]);
        }
    }

    private function seedEssentialData()
    {
        // Seed basic countries
        $countries = [
            ['code' => 'KE', 'name' => 'Kenya', 'region' => 'EAC'],
            ['code' => 'TZ', 'name' => 'Tanzania', 'region' => 'EAC'],
            ['code' => 'UG', 'name' => 'Uganda', 'region' => 'EAC'],
            ['code' => 'CN', 'name' => 'China', 'region' => 'OTHER'],
            ['code' => 'US', 'name' => 'United States', 'region' => 'OTHER'],
            ['code' => 'IN', 'name' => 'India', 'region' => 'OTHER'],
            ['code' => 'GB', 'name' => 'United Kingdom', 'region' => 'OTHER'],
            ['code' => 'DE', 'name' => 'Germany', 'region' => 'OTHER'],
            ['code' => 'JP', 'name' => 'Japan', 'region' => 'OTHER'],
        ];

        foreach ($countries as $country) {
            Country::create($country);
        }

        // Seed currencies
        $currencies = [
            ['code' => 'KES', 'name' => 'Kenyan Shilling', 'symbol' => 'KSh', 'active' => true],
            ['code' => 'USD', 'name' => 'US Dollar', 'symbol' => '$', 'active' => true],
            ['code' => 'EUR', 'name' => 'Euro', 'symbol' => '€', 'active' => true],
            ['code' => 'GBP', 'name' => 'British Pound', 'symbol' => '£', 'active' => true],
        ];

        foreach ($currencies as $currency) {
            Currency::create($currency);
        }

        // Seed tax configurations
        $taxConfigs = [
            [
                'tax_name' => 'VAT',
                'rate_type' => 'percentage',
                'rate' => 16.0,
                'min_amount' => null,
                'active' => true,
                'effective_from' => now(),
                'remarks' => 'Standard VAT rate',
            ],
            [
                'tax_name' => 'IDF',
                'rate_type' => 'percentage',
                'rate' => 3.5,
                'min_amount' => 5000,
                'active' => true,
                'effective_from' => now(),
                'remarks' => 'Import Declaration Fee',
            ],
            [
                'tax_name' => 'RDL',
                'rate_type' => 'percentage',
                'rate' => 2.0,
                'min_amount' => null,
                'active' => true,
                'effective_from' => now(),
                'remarks' => 'Railway Development Levy',
            ],
        ];

        foreach ($taxConfigs as $config) {
            TaxConfig::create($config);
        }

        // Seed sample HS codes
        $hsCodes = [
            [
                'hs_code' => '870321',
                'description' => 'Motor cars for transport of persons',
                'default_duty_type' => 'percentage',
                'default_duty_value' => 25.0,
                'excise_applicable' => true,
                'excise_formula' => json_encode(['type' => 'percentage', 'value' => 20]),
                'sensitive' => false,
            ],
            [
                'hs_code' => '851712',
                'description' => 'Mobile phones',
                'default_duty_type' => 'percentage',
                'default_duty_value' => 10.0,
                'excise_applicable' => true,
                'excise_formula' => json_encode(['type' => 'percentage', 'value' => 10]),
                'sensitive' => false,
            ],
        ];

        foreach ($hsCodes as $hsCode) {
            HsCode::create($hsCode);
        }
    }
}