<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\BaseController;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SettingController extends BaseController
{
    public function index()
    {
        // Get all settings grouped by category
        $settings = Setting::all()->groupBy('category');
        
        $categories = [
            'general' => 'General Settings',
            'contact' => 'Contact Information', 
            'appearance' => 'Appearance',
            'tax' => 'Tax Configuration',
            'notifications' => 'Notifications'
        ];

        return view('admin.settings.index', compact('settings', 'categories'));
    }

    public function updateGeneral(Request $request)
    {
        $validated = $request->validate([
            'company_name' => 'required|string|max:255',
            'company_address' => 'nullable|string|max:500',
            'company_phone' => 'nullable|string|max:20',
            'default_currency' => 'required|string|size:3',
            'default_timezone' => 'required|timezone',
            'date_format' => 'required|string|max:20',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value);
        }

        // Handle logo upload
        if ($request->hasFile('app_logo')) {
            $this->handleLogoUpload($request->file('app_logo'));
        }

        return redirect()->route('admin.settings.index')->with('success', 'General settings updated successfully.');
    }

    public function updateContact(Request $request)
    {
        $validated = $request->validate([
            'contact_email' => 'required|email',
            'whatsapp_number' => 'required|string|max:20',
            'support_phone' => 'nullable|string|max:20',
            'office_address' => 'nullable|string|max:500',
            'business_hours' => 'nullable|string|max:255',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value);
        }

        return redirect()->route('admin.settings.index')->with('success', 'Contact settings updated successfully.');
    }

    public function updateTax(Request $request)
    {
        $validated = $request->validate([
            'default_tax_version' => 'required|string|max:50',
            'calculation_precision' => 'required|integer|min:0|max:8',
            'auto_update_exchange_rates' => 'boolean',
            'exchange_rate_margin' => 'required|numeric|min:0|max:10',
            'enable_benchmarks' => 'boolean',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value);
        }

        return redirect()->route('admin.settings.index')->with('success', 'Tax settings updated successfully.');
    }

    public function updateNotifications(Request $request)
    {
        $validated = $request->validate([
            'admin_email_notifications' => 'boolean',
            'user_email_notifications' => 'boolean',
            'low_stock_threshold' => 'nullable|integer|min:0',
            'system_health_alerts' => 'boolean',
            'backup_reminders' => 'boolean',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value);
        }

        return redirect()->route('admin.settings.index')->with('success', 'Notification settings updated successfully.');
    }

    private function handleLogoUpload($file)
    {
        $validated = validator(['logo' => $file], [
            'logo' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048'
        ])->validate();

        // Delete old logo if exists
        $oldLogo = Setting::get('app_logo');
        if ($oldLogo && Storage::disk('public')->exists($oldLogo)) {
            Storage::disk('public')->delete($oldLogo);
        }

        // Store new logo
        $path = $file->store('logos', 'public');
        Setting::set('app_logo', $path);

        return $path;
    }

    public function removeLogo()
    {
        $logoPath = Setting::get('app_logo');
        if ($logoPath && Storage::disk('public')->exists($logoPath)) {
            Storage::disk('public')->delete($logoPath);
        }

        Setting::set('app_logo', null);

        return redirect()->route('admin.settings.index')->with('success', 'Logo removed successfully.');
    }
}