<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\BaseController;
use App\Models\AdminRole;
use App\Models\User;
use Illuminate\Http\Request;

class RoleController extends BaseController
{
    public function index()
    {
        // Get roles with manual user count
        $roles = AdminRole::all()->map(function($role) {
            $role->users_count = User::where('role_id', $role->id)->count();
            return $role;
        });
        
        return view('admin.roles.index', compact('roles'));
    }

    public function create()
    {
        $permissionGroups = $this->getPermissionGroups();
        return view('admin.roles.create', compact('permissionGroups'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:admin_roles',
            'description' => 'nullable|string',
            'permissions' => 'required|array',
            'is_active' => 'boolean'
        ]);

        $validated['permissions'] = json_encode($validated['permissions']);

        AdminRole::create($validated);

        return $this->successResponse('Role created successfully.', 'admin.roles.index');
    }

    public function edit(AdminRole $role)
    {
        if ($role->slug === 'super_admin') {
            return $this->errorResponse('Super admin role cannot be edited.');
        }

        $permissionGroups = $this->getPermissionGroups();
        return view('admin.roles.edit', compact('role', 'permissionGroups'));
    }

    public function update(Request $request, AdminRole $role)
    {
        if ($role->slug === 'super_admin') {
            return $this->errorResponse('Super admin role cannot be modified.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:admin_roles,slug,' . $role->id,
            'description' => 'nullable|string',
            'permissions' => 'required|array',
            'is_active' => 'boolean'
        ]);

        $validated['permissions'] = json_encode($validated['permissions']);

        $role->update($validated);

        return $this->successResponse('Role updated successfully.', 'admin.roles.index');
    }

    public function destroy(AdminRole $role)
    {
        if ($role->slug === 'super_admin') {
            return $this->errorResponse('Super admin role cannot be deleted.');
        }

        $userCount = User::where('role_id', $role->id)->count();
        if ($userCount > 0) {
            return $this->errorResponse('Cannot delete role that has users assigned.');
        }

        $role->delete();

        return $this->successResponse('Role deleted successfully.', 'admin.roles.index');
    }

    private function getPermissionGroups()
    {
        return [
            'Dashboard' => [
                'dashboard.view' => 'View Dashboard'
            ],
            'Content Management' => [
                'countries.*' => 'Full Countries Access',
                'countries.view' => 'View Countries',
                'countries.create' => 'Create Countries',
                'countries.edit' => 'Edit Countries',
                'countries.delete' => 'Delete Countries',
                'faqs.*' => 'Full FAQs Access',
                'faqs.view' => 'View FAQs',
                'faqs.create' => 'Create FAQs',
                'faqs.edit' => 'Edit FAQs',
                'faqs.delete' => 'Delete FAQs'
            ],
            'Tax Configuration' => [
                'hs-codes.*' => 'Full HS Codes Access',
                'tax-configs.*' => 'Full Tax Configs Access',
                'benchmarks.*' => 'Full Benchmarks Access',
                'country-overrides.*' => 'Full Country Overrides Access'
            ],
            'Financial' => [
                'currencies.*' => 'Full Currencies Access',
                'exchange-rates.*' => 'Full Exchange Rates Access'
            ],
            'User Management' => [
                'users.*' => 'Full Users Access',
                'users.view' => 'View Users',
                'users.create' => 'Create Users',
                'users.edit' => 'Edit Users',
                'users.delete' => 'Delete Users'
            ],
            'Reports & Analytics' => [
                'reports.*' => 'Full Reports Access',
                'reports.view' => 'View Reports',
                'calculation-logs.*' => 'Full Calculation Logs Access',
                'calculation-logs.view' => 'View Calculation Logs'
            ],
            'System' => [
                'audit-logs.*' => 'Full Audit Logs Access',
                'audit-logs.view' => 'View Audit Logs',
                'settings.*' => 'Full Settings Access',
                'settings.view' => 'View Settings',
                'settings.edit' => 'Edit Settings',
                'contact-submissions.*' => 'Full Contact Submissions Access',
                'contact-submissions.view' => 'View Contact Submissions'
            ]
        ];
    }
}