<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\BaseController;
use App\Models\CalculationLog;
use App\Models\HsCode;
use App\Models\Country;
use App\Models\TaxConfig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportController extends BaseController
{
    public function index()
    {
        return view('admin.reports.index');
    }

    public function taxCollection(Request $request)
    {
        $dateRange = $this->getDateRange($request);
        
        $taxCollection = CalculationLog::selectRaw('
            DATE(created_at) as date,
            COUNT(*) as total_calculations,
            SUM(total_payable) as total_revenue,
            AVG(total_payable) as average_tax
        ')
        ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
        ->groupBy('date')
        ->orderBy('date')
        ->get();

        $summary = CalculationLog::selectRaw('
            COUNT(*) as total_calculations,
            SUM(total_payable) as total_revenue,
            AVG(total_payable) as average_tax,
            MAX(total_payable) as highest_tax,
            MIN(total_payable) as lowest_tax
        ')
        ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
        ->first();

        return view('admin.reports.tax-collection', compact(
            'taxCollection', 
            'summary',
            'dateRange'
        ));
    }

    public function hsCodeAnalysis(Request $request)
    {
        $dateRange = $this->getDateRange($request);

        $hsCodeStats = CalculationLog::selectRaw('
            JSON_UNQUOTE(JSON_EXTRACT(input, "$.hs_code")) as hs_code,
            COUNT(*) as calculation_count,
            SUM(total_payable) as total_revenue,
            AVG(total_payable) as average_tax,
            MAX(total_payable) as highest_tax
        ')
        ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
        ->groupBy('hs_code')
        ->orderByDesc('calculation_count')
        ->get()
        ->map(function($item) {
            $hsCode = HsCode::where('hs_code', $item->hs_code)->first();
            return [
                'hs_code' => $item->hs_code,
                'description' => $hsCode->description ?? 'Unknown',
                'calculation_count' => $item->calculation_count,
                'total_revenue' => $item->total_revenue,
                'average_tax' => $item->average_tax,
                'highest_tax' => $item->highest_tax
            ];
        });

        $popularHsCodes = $hsCodeStats->take(10);
        $revenueByHsCode = $hsCodeStats->sortByDesc('total_revenue')->take(10);

        return view('admin.reports.hs-code-analysis', compact(
            'popularHsCodes',
            'revenueByHsCode',
            'dateRange'
        ));
    }

    public function countryAnalysis(Request $request)
    {
        $dateRange = $this->getDateRange($request);

        $countryStats = CalculationLog::selectRaw('
            JSON_UNQUOTE(JSON_EXTRACT(input, "$.country_of_origin")) as country_code,
            COUNT(*) as calculation_count,
            SUM(total_payable) as total_revenue,
            AVG(total_payable) as average_tax
        ')
        ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
        ->groupBy('country_code')
        ->orderByDesc('calculation_count')
        ->get()
        ->map(function($item) {
            $country = Country::where('code', $item->country_code)->first();
            return [
                'country_code' => $item->country_code,
                'country_name' => $country->name ?? 'Unknown',
                'region' => $country->region ?? 'Unknown',
                'calculation_count' => $item->calculation_count,
                'total_revenue' => $item->total_revenue,
                'average_tax' => $item->average_tax
            ];
        });

        $calculationsByCountry = $countryStats->sortByDesc('calculation_count')->take(10);
        $revenueByCountry = $countryStats->sortByDesc('total_revenue')->take(10);
        $calculationsByRegion = $countryStats->groupBy('region')->map(function($items, $region) {
            return [
                'region' => $region,
                'calculation_count' => $items->sum('calculation_count'),
                'total_revenue' => $items->sum('total_revenue')
            ];
        })->sortByDesc('calculation_count');

        return view('admin.reports.country-analysis', compact(
            'calculationsByCountry',
            'revenueByCountry',
            'calculationsByRegion',
            'dateRange'
        ));
    }

    public function generate(Request $request)
    {
        $request->validate([
            'report_type' => 'required|in:tax_collection,hs_code_analysis,country_analysis',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'format' => 'required|in:html,pdf,csv'
        ]);

        $dateRange = [
            'start' => Carbon::parse($request->start_date)->startOfDay(),
            'end' => Carbon::parse($request->end_date)->endOfDay()
        ];

        switch ($request->report_type) {
            case 'tax_collection':
                return $this->generateTaxCollectionReport($dateRange, $request->format);
            case 'hs_code_analysis':
                return $this->generateHsCodeAnalysisReport($dateRange, $request->format);
            case 'country_analysis':
                return $this->generateCountryAnalysisReport($dateRange, $request->format);
        }

        return redirect()->back()->with('error', 'Invalid report type.');
    }

    public function export($type)
    {
        // Simple export functionality - can be enhanced with proper export libraries
        $dateRange = $this->getDateRange(request());

        switch ($type) {
            case 'tax-collection':
                return $this->exportTaxCollection($dateRange);
            case 'hs-code-analysis':
                return $this->exportHsCodeAnalysis($dateRange);
            case 'country-analysis':
                return $this->exportCountryAnalysis($dateRange);
        }

        return redirect()->back()->with('error', 'Invalid export type.');
    }

    private function getDateRange($request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        return [
            'start' => Carbon::parse($startDate)->startOfDay(),
            'end' => Carbon::parse($endDate)->endOfDay()
        ];
    }

    private function generateTaxCollectionReport($dateRange, $format)
    {
        // For now, redirect to HTML view
        // In production, you can implement PDF/CSV generation using DomPDF, Maatwebsite Excel, etc.
        return redirect()->route('admin.reports.tax-collection', [
            'start_date' => $dateRange['start']->format('Y-m-d'),
            'end_date' => $dateRange['end']->format('Y-m-d')
        ]);
    }

    private function generateHsCodeAnalysisReport($dateRange, $format)
    {
        return redirect()->route('admin.reports.hs-code-analysis', [
            'start_date' => $dateRange['start']->format('Y-m-d'),
            'end_date' => $dateRange['end']->format('Y-m-d')
        ]);
    }

    private function generateCountryAnalysisReport($dateRange, $format)
    {
        return redirect()->route('admin.reports.country-analysis', [
            'start_date' => $dateRange['start']->format('Y-m-d'),
            'end_date' => $dateRange['end']->format('Y-m-d')
        ]);
    }

    private function exportTaxCollection($dateRange)
    {
        // Simple CSV export implementation
        $data = CalculationLog::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->select('created_at', 'total_payable', DB::raw('JSON_UNQUOTE(JSON_EXTRACT(input, "$.hs_code")) as hs_code'))
            ->get();

        $filename = "tax-collection-{$dateRange['start']->format('Y-m-d')}-to-{$dateRange['end']->format('Y-m-d')}.csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function() use ($data) {
            $file = fopen('php://output', 'w');
            fputcsv($file, ['Date', 'HS Code', 'Tax Amount', 'Currency']);
            
            foreach ($data as $row) {
                fputcsv($file, [
                    $row->created_at->format('Y-m-d H:i:s'),
                    $row->hs_code,
                    $row->total_payable,
                    'KES'
                ]);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function exportHsCodeAnalysis($dateRange)
    {
        $data = CalculationLog::selectRaw('
            JSON_UNQUOTE(JSON_EXTRACT(input, "$.hs_code")) as hs_code,
            COUNT(*) as calculation_count,
            SUM(total_payable) as total_revenue
        ')
        ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
        ->groupBy('hs_code')
        ->get();

        $filename = "hs-code-analysis-{$dateRange['start']->format('Y-m-d')}-to-{$dateRange['end']->format('Y-m-d')}.csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function() use ($data) {
            $file = fopen('php://output', 'w');
            fputcsv($file, ['HS Code', 'Calculation Count', 'Total Revenue (KES)']);
            
            foreach ($data as $row) {
                fputcsv($file, [
                    $row->hs_code,
                    $row->calculation_count,
                    $row->total_revenue
                ]);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function exportCountryAnalysis($dateRange)
    {
        $data = CalculationLog::selectRaw('
            JSON_UNQUOTE(JSON_EXTRACT(input, "$.country_of_origin")) as country_code,
            COUNT(*) as calculation_count,
            SUM(total_payable) as total_revenue
        ')
        ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
        ->groupBy('country_code')
        ->get();

        $filename = "country-analysis-{$dateRange['start']->format('Y-m-d')}-to-{$dateRange['end']->format('Y-m-d')}.csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function() use ($data) {
            $file = fopen('php://output', 'w');
            fputcsv($file, ['Country Code', 'Calculation Count', 'Total Revenue (KES)']);
            
            foreach ($data as $row) {
                fputcsv($file, [
                    $row->country_code,
                    $row->calculation_count,
                    $row->total_revenue
                ]);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}