<?php

namespace App\Http\Controllers\Admin;

use App\Models\HsCode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class HsCodeController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        $query = HsCode::query();

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->where('hs_code', 'LIKE', "%{$request->search}%")
                  ->orWhere('description', 'LIKE', "%{$request->search}%");
            });
        }

        // Filters
        if ($request->has('duty_type') && $request->duty_type) {
            $query->where('default_duty_type', $request->duty_type);
        }

        if ($request->has('excise_applicable') && $request->excise_applicable !== '') {
            $query->where('excise_applicable', $request->excise_applicable);
        }

        $hsCodes = $query->latest()->paginate($this->perPage);

        return view('admin.hs-codes.index', compact('hsCodes'));
    }

    public function create()
    {
        return view('admin.hs-codes.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'hs_code' => 'required|string|max:20|unique:hs_codes,hs_code',
            'description' => 'required|string|max:500',
            'default_duty_type' => 'required|in:percentage,flat,per_kg,per_container',
            'default_duty_value' => 'required|numeric|min:0',
            'excise_applicable' => 'boolean',
            'excise_formula' => 'nullable|json',
            'sensitive' => 'boolean',
        ]);

        try {
            HsCode::create($validated);
            return redirect()->route('admin.hs-codes.index')
                ->with('success', 'HS Code created successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create HS Code: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(HsCode $hsCode)
    {
        return view('admin.hs-codes.show', compact('hsCode'));
    }

    public function edit(HsCode $hsCode)
    {
        return view('admin.hs-codes.edit', compact('hsCode'));
    }

    public function update(Request $request, HsCode $hsCode)
    {
        $validated = $request->validate([
            'hs_code' => 'required|string|max:20|unique:hs_codes,hs_code,' . $hsCode->id,
            'description' => 'required|string|max:500',
            'default_duty_type' => 'required|in:percentage,flat,per_kg,per_container',
            'default_duty_value' => 'required|numeric|min:0',
            'excise_applicable' => 'boolean',
            'excise_formula' => 'nullable|json',
            'sensitive' => 'boolean',
        ]);

        try {
            $hsCode->update($validated);
            return redirect()->route('admin.hs-codes.index')
                ->with('success', 'HS Code updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update HS Code: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(HsCode $hsCode)
    {
        try {
            // Check if HS code is being used
            if ($hsCode->benchmarks()->exists() || $hsCode->countryOverrides()->exists()) {
                return redirect()->back()
                    ->with('error', 'Cannot delete HS Code. It is being used in benchmarks or country overrides.');
            }

            $hsCode->delete();
            return redirect()->route('admin.hs-codes.index')
                ->with('success', 'HS Code deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete HS Code: ' . $e->getMessage());
        }
    }

    public function bulkImport(Request $request)
    {
        $request->validate([
            'import_file' => 'required|file|mimes:csv,txt,xlsx,xls|max:10240', // 10MB max
        ]);

        try {
            $file = $request->file('import_file');
            $extension = $file->getClientOriginalExtension();
            
            $imported = 0;
            $updated = 0;
            $errors = [];

            if (in_array($extension, ['csv', 'txt'])) {
                $this->importFromCsv($file, $imported, $updated, $errors);
            } elseif (in_array($extension, ['xlsx', 'xls'])) {
                $this->importFromExcel($file, $imported, $updated, $errors);
            } else {
                throw new \Exception('Unsupported file format');
            }

            $message = "Import completed: {$imported} new records, {$updated} updated records.";
            if (!empty($errors)) {
                $message .= ' ' . count($errors) . ' errors occurred.';
            }

            return redirect()->route('admin.hs-codes.index')
                ->with('success', $message)
                ->with('import_errors', $errors);

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    private function importFromCsv($file, &$imported, &$updated, &$errors)
    {
        $handle = fopen($file->getPathname(), 'r');
        $headers = fgetcsv($handle); // Get column headers
        
        $rowNumber = 1;
        while (($row = fgetcsv($handle)) !== FALSE) {
            $rowNumber++;
            
            try {
                $data = array_combine($headers, $row);
                $this->processImportRow($data, $imported, $updated);
            } catch (\Exception $e) {
                $errors[] = "Row {$rowNumber}: " . $e->getMessage();
            }
        }
        
        fclose($handle);
    }

    private function importFromExcel($file, &$imported, &$updated, &$errors)
    {
        // For Excel files, you would use PhpSpreadsheet
        // This is a simplified version - you might want to install and use PhpSpreadsheet
        throw new \Exception('Excel import requires PhpSpreadsheet package. Please use CSV format.');
    }

    private function processImportRow($data, &$imported, &$updated)
    {
        $validator = Validator::make($data, [
            'hs_code' => 'required|string|max:20',
            'description' => 'required|string|max:500',
            'default_duty_type' => 'required|in:percentage,flat,per_kg,per_container',
            'default_duty_value' => 'required|numeric|min:0',
            'excise_applicable' => 'nullable|boolean',
            'sensitive' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            throw new \Exception(implode(', ', $validator->errors()->all()));
        }

        // Process boolean fields
        $data['excise_applicable'] = filter_var($data['excise_applicable'] ?? false, FILTER_VALIDATE_BOOLEAN);
        $data['sensitive'] = filter_var($data['sensitive'] ?? false, FILTER_VALIDATE_BOOLEAN);

        // Check if HS code exists
        $existing = HsCode::where('hs_code', $data['hs_code'])->first();

        if ($existing) {
            // Update existing
            $existing->update($data);
            $updated++;
        } else {
            // Create new
            HsCode::create($data);
            $imported++;
        }
    }

    public function export(Request $request)
    {
        $query = HsCode::query();

        // Apply filters if any
        if ($request->has('duty_type') && $request->duty_type) {
            $query->where('default_duty_type', $request->duty_type);
        }

        if ($request->has('excise_applicable') && $request->excise_applicable !== '') {
            $query->where('excise_applicable', $request->excise_applicable);
        }

        $hsCodes = $query->get();

        $fileName = 'hs_codes_export_' . date('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
        ];

        $callback = function() use ($hsCodes) {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fputs($file, $bom = (chr(0xEF) . chr(0xBB) . chr(0xBF)));
            
            // Headers
            fputcsv($file, [
                'HS Code',
                'Description',
                'Default Duty Type',
                'Default Duty Value',
                'Excise Applicable',
                'Sensitive',
                'Created At',
                'Updated At'
            ]);

            // Data
            foreach ($hsCodes as $hsCode) {
                fputcsv($file, [
                    $hsCode->hs_code,
                    $hsCode->description,
                    $hsCode->default_duty_type,
                    $hsCode->default_duty_value,
                    $hsCode->excise_applicable ? 'Yes' : 'No',
                    $hsCode->sensitive ? 'Yes' : 'No',
                    $hsCode->created_at->format('Y-m-d H:i:s'),
                    $hsCode->updated_at->format('Y-m-d H:i:s')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function downloadTemplate()
    {
        $fileName = 'hs_codes_import_template.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
        ];

        $callback = function() {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fputs($file, $bom = (chr(0xEF) . chr(0xBB) . chr(0xBF)));
            
            // Headers with examples
            fputcsv($file, [
                'hs_code',
                'description', 
                'default_duty_type',
                'default_duty_value',
                'excise_applicable',
                'sensitive'
            ]);

            // Example rows
            fputcsv($file, [
                '870321',
                'Passenger motor vehicles with spark-ignition engine',
                'percentage',
                '25.00',
                '1',
                '0'
            ]);

            fputcsv($file, [
                '030211',
                'Fresh or chilled trout',
                'per_kg',
                '150.00',
                '0',
                '1'
            ]);

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}