<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Faq;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FaqController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $faqs = Faq::orderBy('order')->orderBy('created_at')->get();
        return view('admin.faqs.index', compact('faqs'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Faq::distinct()->pluck('category')->toArray();
        $defaultCategories = ['General', 'Technical', 'Calculation', 'Account', 'Services', 'Payment'];
        $allCategories = array_unique(array_merge($defaultCategories, $categories));
        
        return view('admin.faqs.create', compact('allCategories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'question' => 'required|string|max:255',
            'answer' => 'required|string',
            'category' => 'required|string|max:100',
            'order' => 'nullable|integer',
            'is_published' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        Faq::create([
            'question' => $request->question,
            'answer' => $request->answer,
            'category' => $request->category,
            'order' => $request->order ?? 0,
            'is_published' => $request->has('is_published'),
        ]);

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Faq $faq)
    {
        return view('admin.faqs.show', compact('faq'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Faq $faq)
    {
        $categories = Faq::distinct()->pluck('category')->toArray();
        $defaultCategories = ['General', 'Technical', 'Calculation', 'Account', 'Services', 'Payment'];
        $allCategories = array_unique(array_merge($defaultCategories, $categories));
        
        return view('admin.faqs.edit', compact('faq', 'allCategories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Faq $faq)
    {
        $validator = Validator::make($request->all(), [
            'question' => 'required|string|max:255',
            'answer' => 'required|string',
            'category' => 'required|string|max:100',
            'order' => 'nullable|integer',
            'is_published' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $faq->update([
            'question' => $request->question,
            'answer' => $request->answer,
            'category' => $request->category,
            'order' => $request->order ?? 0,
            'is_published' => $request->has('is_published'),
        ]);

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Faq $faq)
    {
        $faq->delete();

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ deleted successfully!');
    }

    /**
     * Toggle FAQ publish status
     */
    public function toggleStatus(Faq $faq)
    {
        $faq->update([
            'is_published' => !$faq->is_published
        ]);

        $status = $faq->is_published ? 'published' : 'unpublished';
        
        return redirect()->route('admin.faqs.index')
            ->with('success', "FAQ {$status} successfully!");
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $action = $request->input('action');
        $ids = $request->input('ids', []);

        if (empty($ids)) {
            return redirect()->back()->with('error', 'No FAQs selected!');
        }

        switch ($action) {
            case 'publish':
                Faq::whereIn('id', $ids)->update(['is_published' => true]);
                $message = 'Selected FAQs published successfully!';
                break;
            case 'unpublish':
                Faq::whereIn('id', $ids)->update(['is_published' => false]);
                $message = 'Selected FAQs unpublished successfully!';
                break;
            case 'delete':
                Faq::whereIn('id', $ids)->delete();
                $message = 'Selected FAQs deleted successfully!';
                break;
            default:
                return redirect()->back()->with('error', 'Invalid action!');
        }

        return redirect()->route('admin.faqs.index')->with('success', $message);
    }
}