<?php

namespace App\Http\Controllers\Admin;

use App\Models\ExchangeRate;
use App\Models\Currency;
use Illuminate\Http\Request;

class ExchangeRateController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        $query = ExchangeRate::with(['baseCurrency', 'targetCurrency']);

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->whereHas('baseCurrency', function($q) use ($request) {
                    $q->where('code', 'LIKE', "%{$request->search}%")
                      ->orWhere('name', 'LIKE', "%{$request->search}%");
                })->orWhereHas('targetCurrency', function($q) use ($request) {
                    $q->where('code', 'LIKE', "%{$request->search}%")
                      ->orWhere('name', 'LIKE', "%{$request->search}%");
                });
            });
        }

        // Filters
        if ($request->has('base_currency') && $request->base_currency) {
            $query->where('base_currency', $request->base_currency);
        }

        if ($request->has('target_currency') && $request->target_currency) {
            $query->where('target_currency', $request->target_currency);
        }

        if ($request->has('is_active') && $request->is_active !== '') {
            $query->where('is_active', $request->is_active);
        }

        $exchangeRates = $query->latest('effective_date')->paginate($this->perPage);
        $currencies = Currency::active()->orderBy('code')->get();

        return view('admin.exchange-rates.index', compact('exchangeRates', 'currencies'));
    }

    public function create()
    {
        $currencies = Currency::active()->orderBy('code')->get();
        return view('admin.exchange-rates.create', compact('currencies'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'base_currency' => 'required|string|size:3|exists:currencies,code',
            'target_currency' => 'required|string|size:3|exists:currencies,code',
            'rate' => 'required|numeric|min:0',
            'effective_date' => 'required|date',
            'is_active' => 'boolean',
        ]);

        // Check for duplicate
        $existing = ExchangeRate::where('base_currency', $validated['base_currency'])
            ->where('target_currency', $validated['target_currency'])
            ->where('effective_date', $validated['effective_date'])
            ->first();

        if ($existing) {
            return redirect()->back()
                ->with('error', 'An exchange rate already exists for this currency pair and effective date.')
                ->withInput();
        }

        try {
            ExchangeRate::create($validated);
            return redirect()->route('admin.exchange-rates.index')
                ->with('success', 'Exchange rate created successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create exchange rate: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(ExchangeRate $exchangeRate)
    {
        return view('admin.exchange-rates.show', compact('exchangeRate'));
    }

    public function edit(ExchangeRate $exchangeRate)
    {
        $currencies = Currency::active()->orderBy('code')->get();
        return view('admin.exchange-rates.edit', compact('exchangeRate', 'currencies'));
    }

    public function update(Request $request, ExchangeRate $exchangeRate)
    {
        $validated = $request->validate([
            'base_currency' => 'required|string|size:3|exists:currencies,code',
            'target_currency' => 'required|string|size:3|exists:currencies,code',
            'rate' => 'required|numeric|min:0',
            'effective_date' => 'required|date',
            'is_active' => 'boolean',
        ]);

        // Check for duplicate (excluding current record)
        $existing = ExchangeRate::where('base_currency', $validated['base_currency'])
            ->where('target_currency', $validated['target_currency'])
            ->where('effective_date', $validated['effective_date'])
            ->where('id', '!=', $exchangeRate->id)
            ->first();

        if ($existing) {
            return redirect()->back()
                ->with('error', 'Another exchange rate already exists for this currency pair and effective date.')
                ->withInput();
        }

        try {
            $exchangeRate->update($validated);
            return redirect()->route('admin.exchange-rates.index')
                ->with('success', 'Exchange rate updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update exchange rate: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(ExchangeRate $exchangeRate)
    {
        try {
            $exchangeRate->delete();
            return redirect()->route('admin.exchange-rates.index')
                ->with('success', 'Exchange rate deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete exchange rate: ' . $e->getMessage());
        }
    }

    public function quickUpdate(Request $request)
    {
        $validated = $request->validate([
            'base_currency' => 'required|string|size:3',
            'target_currency' => 'required|string|size:3',
            'rate' => 'required|numeric|min:0',
        ]);

        try {
            // Create new exchange rate for today
            ExchangeRate::create([
                'base_currency' => $validated['base_currency'],
                'target_currency' => $validated['target_currency'],
                'rate' => $validated['rate'],
                'effective_date' => now()->format('Y-m-d'),
                'is_active' => true,
            ]);

            return redirect()->route('admin.exchange-rates.index')
                ->with('success', 'Exchange rate updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update exchange rate: ' . $e->getMessage());
        }
    }

    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'rates' => 'required|array',
            'rates.*.base_currency' => 'required|string|size:3',
            'rates.*.target_currency' => 'required|string|size:3',
            'rates.*.rate' => 'required|numeric|min:0',
            'effective_date' => 'required|date',
        ]);

        try {
            foreach ($validated['rates'] as $rateData) {
                ExchangeRate::updateOrCreate(
                    [
                        'base_currency' => $rateData['base_currency'],
                        'target_currency' => $rateData['target_currency'],
                        'effective_date' => $validated['effective_date'],
                    ],
                    [
                        'rate' => $rateData['rate'],
                        'is_active' => true,
                    ]
                );
            }

            return redirect()->route('admin.exchange-rates.index')
                ->with('success', 'Exchange rates updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update exchange rates: ' . $e->getMessage());
        }
    }
}