<?php

namespace App\Http\Controllers;

use App\Models\ContactSubmission;
use App\Models\ContactMessage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class EmailParserController extends Controller
{
    /**
     * Handle incoming email responses via webhook
     */
    public function handleIncomingEmail(Request $request)
    {
        Log::info('Incoming email received', $request->all());

        try {
            // Parse the email content based on your email service provider
            $emailData = $this->parseEmail($request);
            
            if (!$emailData) {
                return response()->json(['error' => 'Could not parse email'], 400);
            }

            // Find the original submission by tracking ID or email thread
            $submission = $this->findRelatedSubmission($emailData);
            
            if (!$submission) {
                Log::warning('No related submission found for email', ['from' => $emailData['from']]);
                return response()->json(['error' => 'No related conversation found'], 404);
            }

            // Add the email response as a new message
            $submission->addMessage('incoming', $emailData['body']);
            
            // Mark as unread and update status
            $submission->markAsUnread();
            $submission->updateStatus('in_progress');

            Log::info('Email response added to conversation', [
                'submission_id' => $submission->id,
                'from' => $emailData['from']
            ]);

            return response()->json(['success' => true, 'submission_id' => $submission->id]);

        } catch (\Exception $e) {
            Log::error('Failed to process incoming email: ' . $e->getMessage());
            return response()->json(['error' => 'Processing failed'], 500);
        }
    }

    /**
     * Parse email data based on different providers
     */
    private function parseEmail($request)
    {
        // Method 1: Mailgun webhook format
        if ($request->has('sender') && $request->has('body-plain')) {
            return [
                'from' => $request->input('sender'),
                'to' => $request->input('recipient'),
                'subject' => $request->input('subject'),
                'body' => $request->input('body-plain'),
                'message_id' => $request->input('Message-Id'),
                'in_reply_to' => $request->input('In-Reply-To'),
            ];
        }

        // Method 2: SendGrid parse API format
        if ($request->has('from') && $request->has('text')) {
            return [
                'from' => $request->input('from'),
                'to' => $request->input('to'),
                'subject' => $request->input('subject'),
                'body' => $request->input('text') ?: $request->input('html'),
                'message_id' => $request->input('message-id'),
            ];
        }

        // Method 3: Postmark webhook format
        if ($request->has('From') && $request->has('TextBody')) {
            return [
                'from' => $request->input('From'),
                'to' => $request->input('To'),
                'subject' => $request->input('Subject'),
                'body' => $request->input('TextBody') ?: $request->input('HtmlBody'),
                'message_id' => $request->input('MessageID'),
            ];
        }

        // Method 4: Generic JSON format
        if ($request->has('from_email') && $request->has('body')) {
            return [
                'from' => $request->input('from_email'),
                'to' => $request->input('to_email'),
                'subject' => $request->input('subject'),
                'body' => $request->input('body'),
                'message_id' => $request->input('message_id'),
            ];
        }

        return null;
    }

    /**
     * Find the related contact submission
     */
    private function findRelatedSubmission($emailData)
    {
        // Method 1: Find by email address
        $email = $this->extractEmail($emailData['from']);
        $submission = ContactSubmission::where('email', $email)
            ->orderBy('created_at', 'desc')
            ->first();

        if ($submission) {
            return $submission;
        }

        // Method 2: Find by subject line (if it contains original subject)
        if (isset($emailData['subject'])) {
            $submission = ContactSubmission::where('subject', 'like', '%' . $emailData['subject'] . '%')
                ->orWhere('email', $email)
                ->orderBy('created_at', 'desc')
                ->first();

            if ($submission) {
                return $submission;
            }
        }

        // Method 3: Create a new submission if no existing one found
        return $this->createNewSubmissionFromEmail($emailData);
    }

    /**
     * Extract email address from string
     */
    private function extractEmail($string)
    {
        preg_match('/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/', $string, $matches);
        return $matches[0] ?? $string;
    }

    /**
     * Extract name from email string
     */
    private function extractName($string)
    {
        if (preg_match('/(.*)<.*>/', $string, $matches)) {
            return trim($matches[1], ' "');
        }
        return explode('@', $this->extractEmail($string))[0];
    }

    /**
     * Create new submission from email (for new conversations)
     */
    private function createNewSubmissionFromEmail($emailData)
    {
        $submission = ContactSubmission::create([
            'name' => $this->extractName($emailData['from']),
            'email' => $this->extractEmail($emailData['from']),
            'subject' => $emailData['subject'] ?? 'Email Response',
            'message' => $emailData['body'],
            'status' => 'new',
            'ip_address' => request()->ip(),
            'user_agent' => 'Email Webhook',
        ]);

        // Add the first message
        $submission->addMessage('incoming', $emailData['body']);

        return $submission;
    }

    /**
     * Test endpoint to verify webhook is working
     */
    public function testWebhook(Request $request)
    {
        $testData = [
            'from' => 'test@example.com',
            'to' => 'responses@yourdomain.com',
            'subject' => 'Re: Test Submission',
            'body' => 'This is a test email response',
        ];

        $request->merge($testData);

        return $this->handleIncomingEmail($request);
    }
}