<?php

namespace App\Http\Controllers\Admin;

use App\Models\Currency;
use App\Models\ExchangeRate;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CurrencyController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        try {
            $query = Currency::query();

            // Search
            if ($request->has('search') && $request->search) {
                $query->where(function($q) use ($request) {
                    $q->where('code', 'LIKE', "%{$request->search}%")
                      ->orWhere('name', 'LIKE', "%{$request->search}%");
                });
            }

            // Filters
            if ($request->has('active') && $request->active !== '') {
                $query->where('active', $request->active);
            }

            $currencies = $query->orderBy('code')->paginate($this->perPage);

            // Get current exchange rates (KES base) - latest rate for each currency
            $currentRates = ExchangeRate::with('targetCurrency')
                ->where('base_currency', 'KES')
                ->where('is_active', true)
                ->where('effective_date', '<=', now())
                ->orderBy('effective_date', 'desc')
                ->get()
                ->groupBy('target_currency')
                ->map(function ($rates) {
                    return $rates->first(); // Get the latest rate for each currency
                })
                ->values(); // Reset keys

            // Get historical rates for the history section
            $historicalQuery = ExchangeRate::with(['baseCurrency', 'targetCurrency'])
                ->where('base_currency', 'KES');
            
            // Filter by currency if provided
            if ($request->has('currency') && $request->currency) {
                $historicalQuery->where('target_currency', $request->currency);
            }
            
            // Filter by date range if provided
            if ($request->has('date_from') && $request->date_from) {
                $historicalQuery->where('effective_date', '>=', $request->date_from);
            }
            
            if ($request->has('date_to') && $request->date_to) {
                $historicalQuery->where('effective_date', '<=', $request->date_to);
            }

            $historicalRates = $historicalQuery->orderBy('effective_date', 'desc')
                ->paginate(10);

            return view('admin.currencies.index', compact(
                'currencies', 
                'currentRates', 
                'historicalRates'
            ));

        } catch (\Exception $e) {
            // Fallback if there are database issues
            return view('admin.currencies.index', [
                'currencies' => collect(),
                'currentRates' => collect(),
                'historicalRates' => collect(),
            ])->with('error', 'Unable to load currency data: ' . $e->getMessage());
        }
    }

    public function create()
    {
        return view('admin.currencies.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|size:3|unique:currencies,code',
            'name' => 'required|string|max:100',
            'symbol' => 'nullable|string|max:10',
            'active' => 'boolean',
            'decimal_places' => 'required|integer|min:0|max:6',
        ]);

        try {
            Currency::create($validated);
            return redirect()->route('admin.currencies.index')
                ->with('success', 'Currency created successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create currency: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Currency $currency)
    {
        $exchangeRates = $currency->exchangeRates()
            ->with('baseCurrency')
            ->latest('effective_date')
            ->paginate(10);
            
        return view('admin.currencies.show', compact('currency', 'exchangeRates'));
    }

    public function edit(Currency $currency)
    {
        return view('admin.currencies.edit', compact('currency'));
    }

    public function update(Request $request, Currency $currency)
    {
        $validated = $request->validate([
            'code' => 'required|string|size:3|unique:currencies,code,' . $currency->id,
            'name' => 'required|string|max:100',
            'symbol' => 'nullable|string|max:10',
            'active' => 'boolean',
            'decimal_places' => 'required|integer|min:0|max:6',
        ]);

        try {
            $currency->update($validated);
            return redirect()->route('admin.currencies.index')
                ->with('success', 'Currency updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update currency: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(Currency $currency)
    {
        try {
            // Check if currency has exchange rates
            if ($currency->exchangeRates()->exists() || $currency->baseExchangeRates()->exists()) {
                return redirect()->back()
                    ->with('error', 'Cannot delete currency. It has exchange rates associated.');
            }

            // Prevent deletion of KES (base currency)
            if ($currency->code === 'KES') {
                return redirect()->back()
                    ->with('error', 'Cannot delete the base currency (KES).');
            }

            $currency->delete();
            return redirect()->route('admin.currencies.index')
                ->with('success', 'Currency deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete currency: ' . $e->getMessage());
        }
    }
}