<?php

namespace App\Http\Controllers\Admin;

use App\Models\CountryDutyOverride;
use App\Models\HsCode;
use App\Models\Country;
use Illuminate\Http\Request;

class CountryOverrideController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        $query = CountryDutyOverride::with(['hsCode', 'country']);

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->whereHas('hsCode', function($q) use ($request) {
                    $q->where('hs_code', 'LIKE', "%{$request->search}%")
                      ->orWhere('description', 'LIKE', "%{$request->search}%");
                })->orWhereHas('country', function($q) use ($request) {
                    $q->where('name', 'LIKE', "%{$request->search}%")
                      ->orWhere('code', 'LIKE', "%{$request->search}%");
                });
            });
        }

        // Filters
        if ($request->has('country_code') && $request->country_code) {
            $query->where('country_code', $request->country_code);
        }

        if ($request->has('duty_type') && $request->duty_type) {
            $query->where('duty_type', $request->duty_type);
        }

        $overrides = $query->latest()->paginate($this->perPage);
        $countries = Country::active()->orderBy('name')->get();

        return view('admin.country-overrides.index', compact('overrides', 'countries'));
    }

    public function create()
    {
        $hsCodes = HsCode::orderBy('hs_code')->get();
        $countries = Country::active()->orderBy('name')->get();
        return view('admin.country-overrides.create', compact('hsCodes', 'countries'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'country_code' => 'required|exists:countries,code',
            'hs_code_id' => 'required|exists:hs_codes,id',
            'duty_type' => 'required|in:percentage,flat,per_kg,per_container',
            'duty_value' => 'required|numeric|min:0',
            'remarks' => 'nullable|string|max:500',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after:effective_from',
        ]);

        // Check for duplicate
        $existing = CountryDutyOverride::where('country_code', $validated['country_code'])
            ->where('hs_code_id', $validated['hs_code_id'])
            ->where('effective_from', $validated['effective_from'])
            ->first();

        if ($existing) {
            return redirect()->back()
                ->with('error', 'A duty override already exists for this country, HS code, and effective date.')
                ->withInput();
        }

        try {
            CountryDutyOverride::create($validated);
            return redirect()->route('admin.country-overrides.index')
                ->with('success', 'Country duty override created successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create country duty override: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(CountryDutyOverride $countryOverride)
    {
        return view('admin.country-overrides.show', compact('countryOverride'));
    }

    public function edit(CountryDutyOverride $countryOverride)
    {
        $hsCodes = HsCode::orderBy('hs_code')->get();
        $countries = Country::active()->orderBy('name')->get();
        return view('admin.country-overrides.edit', compact('countryOverride', 'hsCodes', 'countries'));
    }

    public function update(Request $request, CountryDutyOverride $countryOverride)
    {
        $validated = $request->validate([
            'country_code' => 'required|exists:countries,code',
            'hs_code_id' => 'required|exists:hs_codes,id',
            'duty_type' => 'required|in:percentage,flat,per_kg,per_container',
            'duty_value' => 'required|numeric|min:0',
            'remarks' => 'nullable|string|max:500',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after:effective_from',
        ]);

        // Check for duplicate (excluding current record)
        $existing = CountryDutyOverride::where('country_code', $validated['country_code'])
            ->where('hs_code_id', $validated['hs_code_id'])
            ->where('effective_from', $validated['effective_from'])
            ->where('id', '!=', $countryOverride->id)
            ->first();

        if ($existing) {
            return redirect()->back()
                ->with('error', 'Another duty override already exists for this country, HS code, and effective date.')
                ->withInput();
        }

        try {
            $countryOverride->update($validated);
            return redirect()->route('admin.country-overrides.index')
                ->with('success', 'Country duty override updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update country duty override: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(CountryDutyOverride $countryOverride)
    {
        try {
            $countryOverride->delete();
            return redirect()->route('admin.country-overrides.index')
                ->with('success', 'Country duty override deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete country duty override: ' . $e->getMessage());
        }
    }

    public function clone($id)
    {
        $original = CountryDutyOverride::findOrFail($id);
        $hsCodes = HsCode::orderBy('hs_code')->get();
        $countries = Country::active()->orderBy('name')->get();
        
        return view('admin.country-overrides.clone', compact('original', 'hsCodes', 'countries'));
    }
}