<?php

namespace App\Http\Controllers\Admin;

use App\Models\Country;
use Illuminate\Http\Request;

class CountryController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        $query = Country::query();

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->where('name', 'LIKE', "%{$request->search}%")
                  ->orWhere('code', 'LIKE', "%{$request->search}%");
            });
        }

        // Filters
        if ($request->has('region') && $request->region) {
            $query->where('region', $request->region);
        }

        if ($request->has('active') && $request->active !== '') {
            $query->where('active', $request->active);
        }

        $countries = $query->orderBy('name')->paginate($this->perPage);

        return view('admin.countries.index', compact('countries'));
    }

    public function create()
    {
        return view('admin.countries.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|size:2|unique:countries,code',
            'name' => 'required|string|max:100',
            'region' => 'nullable|string|max:50',
            'active' => 'boolean',
        ]);

        try {
            Country::create($validated);
            return redirect()->route('admin.countries.index')
                ->with('success', 'Country created successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create country: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Country $country)
    {
        return view('admin.countries.show', compact('country'));
    }

    public function edit(Country $country)
    {
        return view('admin.countries.edit', compact('country'));
    }

    public function update(Request $request, Country $country)
    {
        $validated = $request->validate([
            'code' => 'required|string|size:2|unique:countries,code,' . $country->id,
            'name' => 'required|string|max:100',
            'region' => 'nullable|string|max:50',
            'active' => 'boolean',
        ]);

        try {
            $country->update($validated);
            return redirect()->route('admin.countries.index')
                ->with('success', 'Country updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update country: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(Country $country)
    {
        try {
            // Check if country has duty overrides
            if ($country->dutyOverrides()->exists()) {
                return redirect()->back()
                    ->with('error', 'Cannot delete country. It has duty overrides associated.');
            }

            $country->delete();
            return redirect()->route('admin.countries.index')
                ->with('success', 'Country deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete country: ' . $e->getMessage());
        }
    }
}