<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ContactSubmission;
use App\Models\ContactMessage;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\ContactResponseMail;

class ContactSubmissionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $submissions = ContactSubmission::with(['assignedTo', 'messages'])
            ->latest()
            ->paginate(20);
            
        $unreadCount = ContactSubmission::unread()->count();
        $totalCount = ContactSubmission::count();
        $statusCounts = [
            'new' => ContactSubmission::new()->count(),
            'in_progress' => ContactSubmission::inProgress()->count(),
            'resolved' => ContactSubmission::status('resolved')->count(),
            'closed' => ContactSubmission::status('closed')->count(),
        ];

        return view('admin.contact-submissions.index', compact('submissions', 'unreadCount', 'totalCount', 'statusCounts'));
    }

    /**
     * Display the specified resource.
     */
    public function show(ContactSubmission $submission)
    {
        // Mark as read when viewing
        if ($submission->isUnread()) {
            $submission->markAsRead();
        }

        // Load messages with user relationship
        $submission->load(['messages.user', 'assignedTo']);
        
        // Get admin users for assignment
        $adminUsers = User::where('is_admin', true)->get();

        return view('admin.contact-submissions.show', compact('submission', 'adminUsers'));
    }

    /**
     * Mark submission as read
     */
    public function markAsRead(ContactSubmission $submission)
    {
        $submission->markAsRead();

        return redirect()->route('admin.contact-submissions.index')
            ->with('success', 'Submission marked as read!');
    }

    /**
     * Mark submission as unread
     */
    public function markAsUnread(ContactSubmission $submission)
    {
        $submission->markAsUnread();

        return redirect()->route('admin.contact-submissions.index')
            ->with('success', 'Submission marked as unread!');
    }

    /**
     * Send response to submission
     */
    public function sendResponse(Request $request, ContactSubmission $submission)
    {
        $request->validate([
            'response_message' => 'required|string|min:10',
            'send_email' => 'boolean',
        ]);

        try {
            // Add message to conversation
            $message = $submission->addMessage('outgoing', $request->response_message, auth()->id());

            // Update submission status
            if ($submission->status === 'new') {
                $submission->updateStatus('in_progress');
            }

            // Send email if requested
            if ($request->boolean('send_email')) {
                Mail::to($submission->email)
                    ->send(new ContactResponseMail($submission, $request->response_message));
            }

            return redirect()->route('admin.contact-submissions.show', $submission->id)
                ->with('success', 'Response added to conversation!' . ($request->boolean('send_email') ? ' Email sent.' : ''));

        } catch (\Exception $e) {
            \Log::error('Failed to send response: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('error', 'Failed to send response: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Update submission status
     */
    public function updateStatus(Request $request, ContactSubmission $submission)
    {
        $request->validate([
            'status' => 'required|in:new,in_progress,resolved,closed',
        ]);

        $submission->updateStatus($request->status);

        return redirect()->route('admin.contact-submissions.show', $submission->id)
            ->with('success', 'Status updated successfully!');
    }

    /**
     * Assign submission to admin
     */
    public function assign(Request $request, ContactSubmission $submission)
    {
        $request->validate([
            'assigned_to' => 'nullable|exists:users,id',
        ]);

        if ($request->assigned_to) {
            $submission->assignTo($request->assigned_to);
            $message = 'Submission assigned successfully!';
        } else {
            $submission->unassign();
            $message = 'Submission unassigned successfully!';
        }

        return redirect()->route('admin.contact-submissions.show', $submission->id)
            ->with('success', $message);
    }

    /**
     * Add internal note (not sent to user)
     */
    public function addNote(Request $request, ContactSubmission $submission)
    {
        $request->validate([
            'note' => 'required|string|min:5',
        ]);

        $submission->addMessage('outgoing', "[INTERNAL NOTE]\n" . $request->note, auth()->id());

        return redirect()->route('admin.contact-submissions.show', $submission->id)
            ->with('success', 'Internal note added!');
    }

    /**
     * Alternative method using raw email
     */
    public function sendResponseAlternative(Request $request, ContactSubmission $submission)
    {
        $request->validate([
            'response_message' => 'required|string|min:10',
        ]);

        try {
            // Simple email sending using Laravel's Mail::raw()
            Mail::raw($request->response_message, function ($message) use ($submission) {
                $message->to($submission->email)
                        ->subject('Re: ' . $submission->subject);
            });

            // Mark as read and add response note
            $submission->update([
                'read_at' => now(),
                'admin_response' => $request->response_message,
                'responded_at' => now(),
            ]);

            return redirect()->route('admin.contact-submissions.show', $submission->id)
                ->with('success', 'Response sent successfully!');

        } catch (\Exception $e) {
            \Log::error('Failed to send response email: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('error', 'Failed to send response: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ContactSubmission $submission)
    {
        $submission->delete();

        return redirect()->route('admin.contact-submissions.index')
            ->with('success', 'Submission deleted successfully!');
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $action = $request->input('action');
        $ids = $request->input('ids', []);

        if (empty($ids)) {
            return redirect()->back()->with('error', 'No submissions selected!');
        }

        switch ($action) {
            case 'mark-read':
                ContactSubmission::whereIn('id', $ids)->update(['read_at' => now()]);
                $message = 'Selected submissions marked as read!';
                break;
            case 'mark-unread':
                ContactSubmission::whereIn('id', $ids)->update(['read_at' => null]);
                $message = 'Selected submissions marked as unread!';
                break;
            case 'delete':
                ContactSubmission::whereIn('id', $ids)->delete();
                $message = 'Selected submissions deleted successfully!';
                break;
            default:
                return redirect()->back()->with('error', 'Invalid action!');
        }

        return redirect()->route('admin.contact-submissions.index')->with('success', $message);
    }
}