<?php

namespace App\Http\Controllers\Admin;

use App\Models\CalculationLog;
use App\Models\User;
use App\Services\TaxCalculatorService;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CalculationLogController extends BaseController
{
    protected $perPage = 20;
    protected $calculatorService;

    public function __construct(TaxCalculatorService $calculatorService)
    {
        $this->calculatorService = $calculatorService;
    }

    public function index(Request $request)
    {
        $query = CalculationLog::with(['user']);

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->where('input->hs_code', 'LIKE', "%{$request->search}%")
                  ->orWhere('input->description', 'LIKE', "%{$request->search}%")
                  ->orWhere('input->country_of_origin', 'LIKE', "%{$request->search}%")
                  ->orWhereHas('user', function($q) use ($request) {
                      $q->where('name', 'LIKE', "%{$request->search}%")
                        ->orWhere('email', 'LIKE', "%{$request->search}%");
                  });
            });
        }

        // Filters
        if ($request->has('user_id') && $request->user_id) {
            $query->where('calculated_by', $request->user_id);
        }

        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->has('min_amount') && $request->min_amount) {
            $query->where('total_payable', '>=', $request->min_amount);
        }

        if ($request->has('max_amount') && $request->max_amount) {
            $query->where('total_payable', '<=', $request->max_amount);
        }

        $calculationLogs = $query->latest()->paginate($this->perPage);

        // Get filter options
        $users = User::whereHas('calculationLogs')->orderBy('name')->get();

        // Statistics
        $stats = [
            'totalCalculations' => CalculationLog::count(),
            'todayCalculations' => CalculationLog::today()->count(),
            'weekCalculations' => CalculationLog::thisWeek()->count(),
            'monthCalculations' => CalculationLog::thisMonth()->count(),
            'totalTaxAmount' => CalculationLog::sum('total_payable'),
            'averageTaxAmount' => CalculationLog::avg('total_payable'),
        ];

        return view('admin.calculation-logs.index', compact(
            'calculationLogs', 
            'users',
            'stats'
        ));
    }

    public function show($id)
    {
        $calculationLog = CalculationLog::with(['user'])->findOrFail($id);
        
        return view('admin.calculation-logs.show', compact('calculationLog'));
    }

    public function recalculate($id)
    {
        try {
            $calculationLog = CalculationLog::findOrFail($id);
            
            // Perform recalculation using the original input
            $result = $this->calculatorService->calculate($calculationLog->input);
            
            // Update the calculation log with new results
            $calculationLog->update([
                'result' => $result,
                'total_payable' => $result['total_taxes'],
                'calculated_by' => auth()->id(),
            ]);

            return redirect()->route('admin.calculation-logs.show', $calculationLog->id)
                ->with('success', 'Calculation has been successfully recalculated.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to recalculate: ' . $e->getMessage());
        }
    }

    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:calculation_logs,id',
            'confirmation' => 'required|string|in:DELETE_CALCULATIONS',
        ]);

        try {
            $deletedCount = CalculationLog::whereIn('id', $validated['ids'])->delete();

            return redirect()->route('admin.calculation-logs.index')
                ->with('success', "Successfully deleted {$deletedCount} calculation logs.");

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete calculation logs: ' . $e->getMessage());
        }
    }

    public function export($type)
    {
        try {
            $calculationLogs = CalculationLog::with(['user'])
                ->latest()
                ->when($type === 'today', function($query) {
                    return $query->today();
                })
                ->when($type === 'week', function($query) {
                    return $query->thisWeek();
                })
                ->when($type === 'month', function($query) {
                    return $query->thisMonth();
                })
                ->get();

            // In a real application, you would generate CSV or Excel file here
            // For now, we'll just return a success message
            $count = $calculationLogs->count();

            return redirect()->route('admin.calculation-logs.index')
                ->with('success', "Export prepared for {$count} calculation logs ({$type}). In a real application, this would download a file.");

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to export calculation logs: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $calculationLog = CalculationLog::findOrFail($id);
            $calculationLog->delete();

            return redirect()->route('admin.calculation-logs.index')
                ->with('success', 'Calculation log deleted successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete calculation log: ' . $e->getMessage());
        }
    }
}