<?php

namespace App\Http\Controllers\Admin;

use App\Models\Benchmark;
use App\Models\HsCode;
use Illuminate\Http\Request;

class BenchmarkController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        $query = Benchmark::with('hsCode');

        // Search
        if ($request->has('search') && $request->search) {
            $query->whereHas('hsCode', function($q) use ($request) {
                $q->where('hs_code', 'LIKE', "%{$request->search}%")
                  ->orWhere('description', 'LIKE', "%{$request->search}%");
            });
        }

        // Filters
        if ($request->has('min_type') && $request->min_type) {
            $query->where('min_type', $request->min_type);
        }

        $benchmarks = $query->latest()->paginate($this->perPage);

        return view('admin.benchmarks.index', compact('benchmarks'));
    }

    public function create()
    {
        $hsCodes = HsCode::orderBy('hs_code')->get();
        return view('admin.benchmarks.create', compact('hsCodes'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'hs_code_id' => 'required|exists:hs_codes,id',
            'min_type' => 'required|in:cif,per_kg,per_container',
            'min_value' => 'required|numeric|min:0',
            'currency' => 'required|string|size:3',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after:effective_from',
        ]);

        try {
            Benchmark::create($validated);
            return redirect()->route('admin.benchmarks.index')
                ->with('success', 'Benchmark created successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create benchmark: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Benchmark $benchmark)
    {
        return view('admin.benchmarks.show', compact('benchmark'));
    }

    public function edit(Benchmark $benchmark)
    {
        $hsCodes = HsCode::orderBy('hs_code')->get();
        return view('admin.benchmarks.edit', compact('benchmark', 'hsCodes'));
    }

    public function update(Request $request, Benchmark $benchmark)
    {
        $validated = $request->validate([
            'hs_code_id' => 'required|exists:hs_codes,id',
            'min_type' => 'required|in:cif,per_kg,per_container',
            'min_value' => 'required|numeric|min:0',
            'currency' => 'required|string|size:3',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after:effective_from',
        ]);

        try {
            $benchmark->update($validated);
            return redirect()->route('admin.benchmarks.index')
                ->with('success', 'Benchmark updated successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update benchmark: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy(Benchmark $benchmark)
    {
        try {
            $benchmark->delete();
            return redirect()->route('admin.benchmarks.index')
                ->with('success', 'Benchmark deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete benchmark: ' . $e->getMessage());
        }
    }
}