<?php

namespace App\Http\Controllers\Admin;

use App\Models\AuditLog;
use App\Models\User;
use Illuminate\Http\Request;
use Carbon\Carbon;

class AuditLogController extends BaseController
{
    protected $perPage = 20;

    public function index(Request $request)
    {
        $query = AuditLog::with(['user', 'model']);

        // Search
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->where('action', 'LIKE', "%{$request->search}%")
                  ->orWhere('model_type', 'LIKE', "%{$request->search}%")
                  ->orWhere('remarks', 'LIKE', "%{$request->search}%")
                  ->orWhereHas('user', function($q) use ($request) {
                      $q->where('name', 'LIKE', "%{$request->search}%")
                        ->orWhere('email', 'LIKE', "%{$request->search}%");
                  });
            });
        }

        // Filters
        if ($request->has('action') && $request->action) {
            $query->where('action', $request->action);
        }

        if ($request->has('model_type') && $request->model_type) {
            $query->where('model_type', $request->model_type);
        }

        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $auditLogs = $query->latest()->paginate($this->perPage);

        // Get filter options
        $users = User::whereHas('auditLogs')->orderBy('name')->get();
        $modelTypes = AuditLog::distinct()->pluck('model_type');
        $actions = AuditLog::distinct()->pluck('action');

        $stats = [
            'totalLogs' => AuditLog::count(),
            'todayLogs' => AuditLog::today()->count(),
            'uniqueUsers' => User::whereHas('auditLogs')->count(),
            'uniqueModels' => AuditLog::distinct('model_type')->count('model_type'),
        ];

        return view('admin.audit-logs.index', compact(
            'auditLogs', 
            'users', 
            'modelTypes', 
            'actions',
            'stats'
        ));
    }

    public function show($id)
    {
        $auditLog = AuditLog::with(['user', 'model'])->findOrFail($id);
        
        return view('admin.audit-logs.show', compact('auditLog'));
    }

    public function clear(Request $request)
    {
        $validated = $request->validate([
            'older_than' => 'required|integer|min:1',
            'confirmation' => 'required|string|in:DELETE_LOGS',
        ]);

        try {
            $days = $validated['older_than'];
            $cutoffDate = Carbon::now()->subDays($days);
            
            $deletedCount = AuditLog::where('created_at', '<', $cutoffDate)->delete();

            return redirect()->route('admin.audit-logs.index')
                ->with('success', "Successfully deleted {$deletedCount} audit logs older than {$days} days.");
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to clear audit logs: ' . $e->getMessage());
        }
    }
}