class TaxCharts {
    constructor() {
        this.charts = new Map();
        this.init();
    }

    init() {
        this.initRevenueChart();
        this.initHsCodeDistributionChart();
        this.initCountryAnalysisChart();
        this.initTaxBreakdownChart();
    }

    initRevenueChart() {
        const ctx = document.getElementById('revenueChart');
        if (!ctx) return;

        this.charts.set('revenue', new Chart(ctx, {
            type: 'line',
            data: {
                labels: this.getLast12Months(),
                datasets: [{
                    label: 'Total Tax Revenue',
                    data: [],
                    borderColor: '#36a2eb',
                    backgroundColor: 'rgba(54, 162, 235, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'Monthly Tax Revenue'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return 'KES ' + value.toLocaleString('en-KE');
                            }
                        }
                    }
                }
            }
        }));

        this.loadRevenueData();
    }

    initHsCodeDistributionChart() {
        const ctx = document.getElementById('hsCodeChart');
        if (!ctx) return;

        this.charts.set('hscode', new Chart(ctx, {
            type: 'bar',
            data: {
                labels: [],
                datasets: [{
                    label: 'Number of Calculations',
                    data: [],
                    backgroundColor: '#4bc0c0'
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'Top HS Codes by Calculation Volume'
                    }
                }
            }
        }));

        this.loadHsCodeData();
    }

    initCountryAnalysisChart() {
        const ctx = document.getElementById('countryChart');
        if (!ctx) return;

        this.charts.set('country', new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: [],
                datasets: [{
                    data: [],
                    backgroundColor: [
                        '#ff6384', '#36a2eb', '#cc65fe', '#ffce56', '#4bc0c0',
                        '#9966ff', '#ff9f40', '#ff6384', '#c9cbcf', '#77ceff'
                    ]
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'Import Origins by Country'
                    },
                    legend: {
                        position: 'right'
                    }
                }
            }
        }));

        this.loadCountryData();
    }

    initTaxBreakdownChart() {
        const ctx = document.getElementById('taxBreakdownChart');
        if (!ctx) return;

        this.charts.set('taxBreakdown', new Chart(ctx, {
            type: 'pie',
            data: {
                labels: ['Duty', 'VAT', 'IDF', 'RDL', 'Excise'],
                datasets: [{
                    data: [0, 0, 0, 0, 0],
                    backgroundColor: [
                        '#ff6384', '#36a2eb', '#cc65fe', '#ffce56', '#4bc0c0'
                    ]
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'Tax Revenue Breakdown'
                    }
                }
            }
        }));

        this.loadTaxBreakdownData();
    }

    async loadRevenueData() {
        try {
            const response = await $.get('/admin/reports/revenue-data');
            const chart = this.charts.get('revenue');
            if (chart && response.data) {
                chart.data.datasets[0].data = response.data;
                chart.update();
            }
        } catch (error) {
            console.error('Failed to load revenue data:', error);
        }
    }

    async loadHsCodeData() {
        try {
            const response = await $.get('/admin/reports/hscode-data');
            const chart = this.charts.get('hscode');
            if (chart && response.data) {
                chart.data.labels = response.data.labels;
                chart.data.datasets[0].data = response.data.values;
                chart.update();
            }
        } catch (error) {
            console.error('Failed to load HS code data:', error);
        }
    }

    async loadCountryData() {
        try {
            const response = await $.get('/admin/reports/country-data');
            const chart = this.charts.get('country');
            if (chart && response.data) {
                chart.data.labels = response.data.labels;
                chart.data.datasets[0].data = response.data.values;
                chart.update();
            }
        } catch (error) {
            console.error('Failed to load country data:', error);
        }
    }

    async loadTaxBreakdownData() {
        try {
            const response = await $.get('/admin/reports/tax-breakdown-data');
            const chart = this.charts.get('taxBreakdown');
            if (chart && response.data) {
                chart.data.datasets[0].data = response.data;
                chart.update();
            }
        } catch (error) {
            console.error('Failed to load tax breakdown data:', error);
        }
    }

    getLast12Months() {
        const months = [];
        for (let i = 11; i >= 0; i--) {
            const date = new Date();
            date.setMonth(date.getMonth() - i);
            months.push(date.toLocaleString('default', { month: 'short', year: '2-digit' }));
        }
        return months;
    }

    // Method to update specific chart
    updateChart(chartName, data) {
        const chart = this.charts.get(chartName);
        if (chart) {
            chart.data = data;
            chart.update();
        }
    }

    // Method to destroy all charts
    destroy() {
        this.charts.forEach(chart => chart.destroy());
        this.charts.clear();
    }
}

// Initialize charts when document is ready
$(document).ready(function() {
    window.taxCharts = new TaxCharts();
});

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = TaxCharts;
}