class TaxCalculator {
    constructor() {
        this.exchangeRates = {};
        this.init();
    }

    init() {
        this.loadExchangeRates();
        this.setupEventListeners();
        this.setupRealTimeValidation();
        this.autoCalculateCIF();
    }

    loadExchangeRates() {
        // Load current exchange rates
        $.get('/api/v1/exchange-rates', (data) => {
            this.exchangeRates = data.rates || {};
        }).fail(() => {
            console.warn('Failed to load exchange rates');
        });
    }

    setupEventListeners() {
        // Form submission
        $('#tax-calculation-form').on('submit', (e) => this.handleCalculation(e));

        // Currency change
        $('#currency').on('change', () => this.updateCurrencyDisplay());

        // HS code lookup
        $('#hs_code').on('blur', () => this.lookupHsCode());

        // Real-time CIF calculation
        $('#customs_value, #freight, #insurance').on('input', () => this.autoCalculateCIF());

        // Container type affects calculations
        $('#container_type').on('change', () => this.toggleWeightField());

        // Country change for duty overrides
        $('#country_of_origin').on('change', () => this.checkCountryOverrides());
    }

    setupRealTimeValidation() {
        // Real-time validation rules
        $('input[type="number"]').on('input', function() {
            const value = parseFloat($(this).val()) || 0;
            const min = parseFloat($(this).attr('min')) || 0;
            
            if (value < min) {
                $(this).addClass('is-invalid');
            } else {
                $(this).removeClass('is-invalid');
            }
        });
    }

    autoCalculateCIF() {
        const customsValue = parseFloat($('#customs_value').val()) || 0;
        const freight = parseFloat($('#freight').val()) || 0;
        const insurance = parseFloat($('#insurance').val()) || 0;
        
        const cif = customsValue + freight + insurance;
        $('#cif_display').text(cif.toLocaleString('en-KE', { 
            style: 'currency', 
            currency: 'KES' 
        }));
    }

    toggleWeightField() {
        const containerType = $('#container_type').val();
        const weightGroup = $('#weight_group');
        
        if (containerType) {
            weightGroup.hide();
        } else {
            weightGroup.show();
        }
    }

    async lookupHsCode() {
        const hsCode = $('#hs_code').val().trim();
        
        if (hsCode.length < 4) return;

        try {
            const response = await $.get(`/api/v1/hs-codes/${hsCode}`);
            if (response.success) {
                this.displayHsCodeInfo(response.data);
            } else {
                this.clearHsCodeInfo();
            }
        } catch (error) {
            this.clearHsCodeInfo();
        }
    }

    displayHsCodeInfo(hsCode) {
        $('#hs_code_description').text(hsCode.description).show();
        $('#duty_type_display').text(this.formatDutyType(hsCode.default_duty_type, hsCode.default_duty_value)).show();
        
        if (hsCode.excise_applicable) {
            $('#excise_notice').show();
        } else {
            $('#excise_notice').hide();
        }
    }

    clearHsCodeInfo() {
        $('#hs_code_description').hide();
        $('#duty_type_display').hide();
        $('#excise_notice').hide();
    }

    formatDutyType(type, value) {
        switch (type) {
            case 'percentage':
                return `${value}% of CIF value`;
            case 'flat':
                return `Flat rate: KES ${value}`;
            case 'per_kg':
                return `KES ${value} per kg`;
            case 'per_container':
                return `KES ${value} per container`;
            default:
                return 'Unknown';
        }
    }

    async checkCountryOverrides() {
        const countryCode = $('#country_of_origin').val();
        const hsCode = $('#hs_code').val().trim();
        
        if (!countryCode || !hsCode) return;

        try {
            const response = await $.get(`/api/v1/country-overrides/${countryCode}/${hsCode}`);
            if (response.success && response.data) {
                this.displayCountryOverrideInfo(response.data);
            } else {
                this.clearCountryOverrideInfo();
            }
        } catch (error) {
            this.clearCountryOverrideInfo();
        }
    }

    displayCountryOverrideInfo(override) {
        $('#country_override_info')
            .text(`Special rate: ${this.formatDutyType(override.duty_type, override.duty_value)}`)
            .addClass('text-success')
            .show();
    }

    clearCountryOverrideInfo() {
        $('#country_override_info').hide().removeClass('text-success');
    }

    updateCurrencyDisplay() {
        const currency = $('#currency').val();
        $('.currency-symbol').text(currency);
    }

    async handleCalculation(e) {
        e.preventDefault();
        
        if (!this.validateForm()) {
            this.showError('Please fill in all required fields correctly.');
            return;
        }

        this.showLoading();

        try {
            const formData = new FormData(e.target);
            const response = await $.ajax({
                url: e.target.action,
                method: 'POST',
                data: formData,
                processData: false,
                contentType: false
            });

            this.displayResults(response);
        } catch (error) {
            this.handleCalculationError(error);
        } finally {
            this.hideLoading();
        }
    }

    validateForm() {
        let isValid = true;
        
        // Required fields
        const requiredFields = ['hs_code', 'country_of_origin', 'customs_value', 'currency'];
        requiredFields.forEach(field => {
            const value = $(`#${field}`).val().trim();
            if (!value) {
                $(`#${field}`).addClass('is-invalid');
                isValid = false;
            } else {
                $(`#${field}`).removeClass('is-invalid');
            }
        });

        // Validate HS code format
        const hsCode = $('#hs_code').val().trim();
        if (hsCode && !/^\d{4,10}$/.test(hsCode)) {
            $('#hs_code').addClass('is-invalid');
            isValid = false;
        }

        return isValid;
    }

    showLoading() {
        $('#calculate-btn')
            .prop('disabled', true)
            .html('<i class="fas fa-spinner fa-spin"></i> Calculating...');
        
        $('#calculation-results').hide();
        $('#calculation-error').hide();
    }

    hideLoading() {
        $('#calculate-btn')
            .prop('disabled', false)
            .html('<i class="fas fa-calculator"></i> Calculate Tax');
    }

    displayResults(results) {
        const template = $('#results-template').html();
        const html = this.renderResultsTemplate(template, results);
        
        $('#calculation-results').html(html).show();
        
        // Scroll to results
        $('html, body').animate({
            scrollTop: $('#calculation-results').offset().top - 100
        }, 500);

        // Update results history if available
        this.updateCalculationHistory(results);
    }

    renderResultsTemplate(template, results) {
        return template
            .replace(/{{total_payable}}/g, this.formatCurrency(results.total_payable))
            .replace(/{{tax_base}}/g, this.formatCurrency(results.tax_base))
            .replace(/{{cif_value}}/g, this.formatCurrency(results.cif_value))
            .replace(/{{benchmark_applied}}/g, results.benchmark_applied ? 'Yes' : 'No')
            .replace(/{{duty_amount}}/g, this.formatCurrency(results.breakdown?.duty))
            .replace(/{{vat_amount}}/g, this.formatCurrency(results.breakdown?.vat))
            .replace(/{{idf_amount}}/g, this.formatCurrency(results.breakdown?.idf))
            .replace(/{{rdl_amount}}/g, this.formatCurrency(results.breakdown?.rdl))
            .replace(/{{excise_amount}}/g, this.formatCurrency(results.breakdown?.excise))
            .replace(/{{calculation_id}}/g, results.calculation_id || '')
            .replace(/{{timestamp}}/g, new Date().toLocaleString('en-KE'));
    }

    formatCurrency(amount) {
        return parseFloat(amount || 0).toLocaleString('en-KE', {
            style: 'currency',
            currency: 'KES',
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }

    handleCalculationError(error) {
        let errorMessage = 'Calculation failed. Please try again.';
        
        if (error.responseJSON) {
            if (error.responseJSON.errors) {
                errorMessage = Object.values(error.responseJSON.errors).join('<br>');
            } else if (error.responseJSON.message) {
                errorMessage = error.responseJSON.message;
            }
        }
        
        this.showError(errorMessage);
    }

    showError(message) {
        $('#calculation-error')
            .html(`<div class="alert alert-danger">${message}</div>`)
            .show();
    }

    updateCalculationHistory(results) {
        // Add to recent calculations list
        const historyList = $('#calculation-history');
        if (historyList.length) {
            const historyItem = `
                <div class="list-group-item">
                    <div class="d-flex w-100 justify-content-between">
                        <h6 class="mb-1">${$('#hs_code').val()} - ${$('#country_of_origin').val()}</h6>
                        <small>${this.formatCurrency(results.total_payable)}</small>
                    </div>
                    <p class="mb-1">${$('#hs_code_description').text()}</p>
                    <small class="text-muted">${new Date().toLocaleString()}</small>
                </div>
            `;
            historyList.prepend(historyItem);
        }
    }

    // Utility method to convert to base currency (KES)
    async convertToKES(amount, fromCurrency) {
        if (fromCurrency === 'KES') return amount;
        
        if (!this.exchangeRates[fromCurrency]) {
            await this.loadExchangeRates();
        }
        
        const rate = this.exchangeRates[fromCurrency];
        return rate ? amount * rate : amount;
    }
}

// Initialize calculator when document is ready
$(document).ready(function() {
    window.taxCalculator = new TaxCalculator();
});

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = TaxCalculator;
}