class TaxSimulatorAdmin {
    constructor() {
        this.init();
    }

    init() {
        this.initDataTables();
        this.initFormValidations();
        this.initAjaxHandlers();
        this.initCalculators();
        this.initBulkOperations();
        this.initDatePickers();
    }

    initDataTables() {
        // Initialize DataTables if available
        if (typeof $.fn.DataTable !== 'undefined') {
            $('.datatable').DataTable({
                pageLength: 25,
                responsive: true,
                autoWidth: false,
                language: {
                    search: "_INPUT_",
                    searchPlaceholder: "Search...",
                    lengthMenu: "_MENU_ records per page",
                    info: "Showing _START_ to _END_ of _TOTAL_ entries",
                    infoEmpty: "Showing 0 to 0 of 0 entries",
                    infoFiltered: "(filtered from _MAX_ total entries)"
                }
            });
        }
    }

    initFormValidations() {
        // Custom validation for HS codes
        $.validator.addMethod("hscode", function(value, element) {
            return this.optional(element) || /^\d{4,10}$/.test(value);
        }, "Please enter a valid HS code (4-10 digits)");

        // Custom validation for percentages
        $.validator.addMethod("percentage", function(value, element) {
            return this.optional(element) || (value >= 0 && value <= 100);
        }, "Please enter a valid percentage (0-100)");

        // Initialize form validation
        $('form.validate').validate({
            errorClass: 'is-invalid',
            validClass: 'is-valid',
            errorElement: 'div',
            errorPlacement: function(error, element) {
                error.addClass('invalid-feedback');
                element.closest('.form-group').append(error);
            },
            highlight: function(element, errorClass, validClass) {
                $(element).addClass(errorClass).removeClass(validClass);
            },
            unhighlight: function(element, errorClass, validClass) {
                $(element).removeClass(errorClass).addClass(validClass);
            }
        });
    }

    initAjaxHandlers() {
        // CSRF token setup for AJAX
        $.ajaxSetup({
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        });

        // Handle AJAX errors
        $(document).ajaxError(function(event, jqXHR, ajaxSettings, thrownError) {
            let message = 'An error occurred while processing your request.';
            
            if (jqXHR.responseJSON && jqXHR.responseJSON.message) {
                message = jqXHR.responseJSON.message;
            }
            
            TaxSimulatorAdmin.showAlert(message, 'danger');
        });
    }

    initCalculators() {
        // Tax calculator functionality
        $('#tax-calculator-form').on('submit', function(e) {
            e.preventDefault();
            TaxSimulatorAdmin.calculateTax($(this));
        });

        // Real-time duty calculation
        $('.duty-calculator').on('input', function() {
            TaxSimulatorAdmin.calculateDutyRealTime();
        });
    }

    initBulkOperations() {
        // Bulk import file handling
        $('#bulk-import-file').on('change', function(e) {
            const fileName = e.target.files[0]?.name;
            if (fileName) {
                $('.file-name').text(fileName);
            }
        });

        // Bulk action confirmation
        $('.bulk-action-btn').on('click', function(e) {
            const action = $(this).data('action');
            const selected = $('.bulk-checkbox:checked').length;
            
            if (selected === 0) {
                e.preventDefault();
                TaxSimulatorAdmin.showAlert('Please select at least one item.', 'warning');
                return false;
            }

            if (action === 'delete') {
                return confirm(`Are you sure you want to delete ${selected} item(s)?`);
            }
        });
    }

    initDatePickers() {
        // Initialize date pickers
        $('.datepicker').daterangepicker({
            singleDatePicker: true,
            showDropdowns: true,
            minYear: 2020,
            maxYear: parseInt(moment().format('YYYY')) + 10,
            locale: {
                format: 'YYYY-MM-DD'
            }
        });

        // Date range pickers
        $('.daterange').daterangepicker({
            autoUpdateInput: false,
            locale: {
                format: 'YYYY-MM-DD',
                cancelLabel: 'Clear'
            }
        });

        $('.daterange').on('apply.daterangepicker', function(ev, picker) {
            $(this).val(picker.startDate.format('YYYY-MM-DD') + ' to ' + picker.endDate.format('YYYY-MM-DD'));
        });

        $('.daterange').on('cancel.daterangepicker', function(ev, picker) {
            $(this).val('');
        });
    }

    static calculateTax(form) {
        const formData = new FormData(form[0]);
        const submitBtn = form.find('button[type="submit"]');
        const originalText = submitBtn.html();

        submitBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Calculating...');

        $.ajax({
            url: form.attr('action'),
            method: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                TaxSimulatorAdmin.displayTaxResults(response);
                submitBtn.prop('disabled', false).html(originalText);
            },
            error: function(xhr) {
                let errorMessage = 'Calculation failed. Please try again.';
                if (xhr.responseJSON && xhr.responseJSON.errors) {
                    errorMessage = Object.values(xhr.responseJSON.errors).join('<br>');
                }
                TaxSimulatorAdmin.showAlert(errorMessage, 'danger');
                submitBtn.prop('disabled', false).html(originalText);
            }
        });
    }

    static displayTaxResults(results) {
        const resultsContainer = $('#tax-results');
        const template = $('#tax-results-template').html();
        
        const html = template
            .replace(/{{total_payable}}/g, results.total_payable?.toLocaleString('en-KE', { style: 'currency', currency: 'KES' }) || '0.00')
            .replace(/{{duty_amount}}/g, results.breakdown?.duty?.toLocaleString('en-KE', { style: 'currency', currency: 'KES' }) || '0.00')
            .replace(/{{vat_amount}}/g, results.breakdown?.vat?.toLocaleString('en-KE', { style: 'currency', currency: 'KES' }) || '0.00')
            .replace(/{{idf_amount}}/g, results.breakdown?.idf?.toLocaleString('en-KE', { style: 'currency', currency: 'KES' }) || '0.00')
            .replace(/{{rdl_amount}}/g, results.breakdown?.rdl?.toLocaleString('en-KE', { style: 'currency', currency: 'KES' }) || '0.00')
            .replace(/{{excise_amount}}/g, results.breakdown?.excise?.toLocaleString('en-KE', { style: 'currency', currency: 'KES' }) || '0.00');

        resultsContainer.html(html).show();
        
        // Scroll to results
        $('html, body').animate({
            scrollTop: resultsContainer.offset().top - 100
        }, 500);
    }

    static showAlert(message, type = 'info') {
        const alertClass = {
            'success': 'alert-success',
            'danger': 'alert-danger',
            'warning': 'alert-warning',
            'info': 'alert-info'
        }[type] || 'alert-info';

        const alertHtml = `
            <div class="alert ${alertClass} alert-dismissible fade show" role="alert">
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                ${message}
            </div>
        `;

        $('#alert-container').html(alertHtml);
        
        // Auto-remove alert after 5 seconds
        setTimeout(() => {
            $('.alert').alert('close');
        }, 5000);
    }

    static confirmAction(message, callback) {
        if (confirm(message)) {
            callback();
        }
    }

    static updateExchangeRates() {
        $.ajax({
            url: '/admin/exchange-rates/update-latest',
            method: 'POST',
            success: function(response) {
                TaxSimulatorAdmin.showAlert('Exchange rates updated successfully!', 'success');
                // Update rates display
                if (response.rates) {
                    Object.keys(response.rates).forEach(currency => {
                        $(`#rate-${currency}`).text(response.rates[currency]);
                    });
                }
            },
            error: function(xhr) {
                TaxSimulatorAdmin.showAlert('Failed to update exchange rates.', 'danger');
            }
        });
    }
}

// Initialize when document is ready
$(document).ready(function() {
    window.adminApp = new TaxSimulatorAdmin();
    
    // Global error handler
    window.addEventListener('error', function(e) {
        console.error('Global error:', e.error);
    });

    // Handle session timeout
    $(document).ajaxComplete(function(event, xhr, settings) {
        if (xhr.status === 419) {
            TaxSimulatorAdmin.showAlert('Your session has expired. Please refresh the page.', 'warning');
        }
    });
});