// Main Suppliers Management System
class SuppliersApp {
    constructor() {
        this.modules = {};
        this.init();
    }

    init() {
        // Check if we're on a suppliers page
        if (!document.getElementById('service-providers-section')) {
            console.log('Suppliers management system not found on this page');
            return;
        }

        console.log('Initializing Suppliers Management System...');
        this.initializeModules();
        this.setupEventListeners();
    }

    initializeModules() {
        try {
            // Define module classes in dependency order
            const moduleClasses = [
                { name: 'searchManager', Class: SearchManager },
                { name: 'exportManager', Class: ExportManager },
                { name: 'modalManager', Class: ModalManager },
                { name: 'vendorManager', Class: VendorManager }
            ];

            moduleClasses.forEach(({ name, Class }) => {
                if (typeof Class !== 'undefined') {
                    try {
                        this.modules[name] = new Class(this);
                        if (typeof this.modules[name].init === 'function') {
                            this.modules[name].init();
                        }
                        console.log(`✅ ${name} initialized successfully`);
                    } catch (error) {
                        console.error(`❌ Error initializing ${name}:`, error);
                    }
                } else {
                    console.warn(`⚠️ Module class ${name} is not defined`);
                }
            });

        } catch (error) {
            console.error('Error initializing modules:', error);
            this.showAlert('Error initializing suppliers system. Some features may not work.', 'error');
        }
    }

    setupEventListeners() {
        // Toggle between Service Providers and Suppliers
        this.setupToggleFunctionality();
    }

    setupToggleFunctionality() {
        const serviceProvidersRadio = document.getElementById('service-providers');
        const suppliersRadio = document.getElementById('suppliers');
        const serviceProvidersSection = document.getElementById('service-providers-section');
        const suppliersSection = document.getElementById('suppliers-section');

        if (serviceProvidersRadio && suppliersRadio) {
            serviceProvidersRadio.addEventListener('change', () => {
                if (serviceProvidersRadio.checked) {
                    serviceProvidersSection.classList.remove('d-none');
                    suppliersSection.classList.add('d-none');
                }
            });

            suppliersRadio.addEventListener('change', () => {
                if (suppliersRadio.checked) {
                    suppliersSection.classList.remove('d-none');
                    serviceProvidersSection.classList.add('d-none');
                }
            });
        }
    }

    getModule(moduleName) {
        return this.modules[moduleName];
    }

    showAlert(message, type = 'info') {
        const alertClass = {
            'success': 'alert-success',
            'error': 'alert-danger',
            'warning': 'alert-warning',
            'info': 'alert-info'
        }[type] || 'alert-info';

        // Remove existing alerts
        document.querySelectorAll('.suppliers-alert').forEach(alert => alert.remove());

        // Create new alert
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert ${alertClass} alert-dismissible fade show suppliers-alert`;
        alertDiv.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;

        const container = document.querySelector('.page-title-box');
        if (container) {
            container.parentNode.insertBefore(alertDiv, container.nextSibling);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                if (alertDiv.parentElement) {
                    alertDiv.remove();
                }
            }, 5000);
        }
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Check if Bootstrap is available
    if (typeof bootstrap === 'undefined') {
        console.error('Bootstrap is not loaded. Please make sure Bootstrap is included on the page.');
        return;
    }

    // Initialize the suppliers app
    try {
        window.suppliersApp = new SuppliersApp();
        console.log('🎉 Suppliers Management System loaded successfully!');
    } catch (error) {
        console.error('Failed to initialize Suppliers App:', error);
    }
});
document.addEventListener('DOMContentLoaded', function() {
    // Wait for app to initialize
    setTimeout(() => {
        if (window.suppliersApp) {
            window.vendorManager = window.suppliersApp.getModule('vendorManager');
        }
    }, 100);
});