// Vendor Management Module
class VendorManager {
    constructor(suppliersApp) {
        this.suppliersApp = suppliersApp;
        this.currentVendorType = 'service-providers';
    }

    init() {
        this.setupEventListeners();
    }

    setupEventListeners() {
        // Add vendor buttons
        document.getElementById('addServiceProviderBtn')?.addEventListener('click', 
            () => this.showAddVendorModal('service-provider'));
        
        document.getElementById('addSupplierBtn')?.addEventListener('click', 
            () => this.showAddVendorModal('supplier'));

        // Table action buttons (delegated event handling)
        document.addEventListener('click', (e) => {
            const button = e.target.closest('[data-action]');
            if (!button) return;

            const action = button.getAttribute('data-action');
            const vendorId = button.getAttribute('data-vendor-id');
            const vendorType = button.closest('[data-vendor-type]')?.getAttribute('data-vendor-type') || 'service-provider';

            switch (action) {
                case 'view':
                    this.viewVendor(vendorId, vendorType);
                    break;
                case 'edit':
                    this.editVendor(vendorId, vendorType);
                    break;
                case 'delete':
                    this.deleteVendor(vendorId);
                    break;
                case 'payment':
                    this.makePayment(vendorId);
                    break;
                case 'summary':
                    this.generateSummary(vendorId);
                    break;
                case 'routes':
                    this.manageRoutes(vendorId);
                    break;
                case 'toggle-status':
                    this.toggleStatus(vendorId);
                    break;
            }
        });

        // Refresh button
        document.querySelectorAll('.btn-outline-secondary').forEach(btn => {
            if (btn.textContent.includes('Refresh')) {
                btn.addEventListener('click', () => this.refreshPage());
            }
        });
    }

    showAddVendorModal(type) {
        const modalManager = this.suppliersApp.getModule('modalManager');
        if (modalManager) {
            modalManager.showAddVendorModal(type);
        }
    }

    editVendor(vendorId, type = 'service-provider') {
        const modalManager = this.suppliersApp.getModule('modalManager');
        if (modalManager) {
            modalManager.showAddVendorModal(type, vendorId);
        }
    }

    viewVendor(vendorId, type) {
        window.location.href = `/service-providers/${vendorId}`;
    }

    makePayment(vendorId) {
        const modalManager = this.suppliersApp.getModule('modalManager');
        if (modalManager) {
            const vendorName = this.getVendorNameFromRow(vendorId);
            modalManager.showPaymentModal(vendorId, vendorName);
        }
    }

    getVendorNameFromRow(vendorId) {
        const row = document.querySelector(`[data-vendor-id="${vendorId}"]`);
        return row ? row.querySelector('h6')?.textContent || 'Vendor' : 'Vendor';
    }

    generateSummary(vendorId) {
        window.location.href = `/service-providers/${vendorId}/summary`;
    }

    manageRoutes(vendorId) {
        const modalManager = this.suppliersApp.getModule('modalManager');
        if (modalManager) {
            modalManager.showRouteModal(vendorId);
        }
    }

    refreshPage() {
        window.location.reload();
    }

    async deleteVendor(vendorId) {
        if (!confirm('Are you sure you want to delete this vendor? This action cannot be undone.')) {
            return;
        }

        try {
            const response = await fetch(`/service-providers/${vendorId}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'Content-Type': 'application/json'
                }
            });

            if (response.ok) {
                this.suppliersApp.showAlert('Vendor deleted successfully', 'success');
                // Remove the row from the table
                const row = document.querySelector(`[data-vendor-id="${vendorId}"]`);
                if (row) row.remove();
                
                // Check if table is empty and show message
                this.checkEmptyTable();
            } else {
                throw new Error('Failed to delete vendor');
            }
        } catch (error) {
            console.error('Error deleting vendor:', error);
            this.suppliersApp.showAlert('Error deleting vendor: ' + error.message, 'error');
        }
    }

    checkEmptyTable() {
        const currentSection = document.getElementById('service-providers-section').classList.contains('d-none') ? 
            'suppliers-section' : 'service-providers-section';
        
        const table = document.querySelector(`#${currentSection} table tbody`);
        if (table && table.children.length === 0) {
            table.innerHTML = `
                <tr>
                    <td colspan="8" class="text-center py-4 text-muted">
                        <i class="ri-inbox-line display-4 d-block mb-2"></i>
                        No vendors found
                    </td>
                </tr>
            `;
        }
    }

    async toggleStatus(vendorId) {
        if (!confirm('Are you sure you want to toggle the status of this vendor?')) {
            return;
        }

        try {
            const response = await fetch(`/service-providers/${vendorId}/toggle-status`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'Content-Type': 'application/json'
                }
            });

            const result = await response.json();
            
            if (result.success) {
                this.suppliersApp.showAlert('Status updated successfully', 'success');
                // Update the status badge in the table
                this.updateStatusBadge(vendorId, result.status);
            } else {
                throw new Error(result.message);
            }
        } catch (error) {
            console.error('Error toggling status:', error);
            this.suppliersApp.showAlert('Error updating status', 'error');
        }
    }

    updateStatusBadge(vendorId, newStatus) {
        const row = document.querySelector(`[data-vendor-id="${vendorId}"]`);
        if (row) {
            const badge = row.querySelector('.badge');
            if (badge) {
                badge.textContent = newStatus.charAt(0).toUpperCase() + newStatus.slice(1);
                badge.className = newStatus === 'active' ? 'badge bg-success' : 'badge bg-danger';
                row.setAttribute('data-status', newStatus);
            }
        }
    }
}