// Modal Management Module
class ModalManager {
    constructor(suppliersApp) {
        this.suppliersApp = suppliersApp;
        this.modals = {};
        this.isEditMode = false;
        this.currentEditId = null;
    }

    init() {
        this.initializeModals();
        this.setupModalEventListeners();
    }

    initializeModals() {
        // Initialize Bootstrap modals
        const modalElements = {
            'serviceProviderModal': 'serviceProviderModal',
            'supplierModal': 'supplierModal',
            'routeModal': 'routeModal',
            'paymentModal': 'paymentModal'
        };

        Object.entries(modalElements).forEach(([key, id]) => {
            const element = document.getElementById(id);
            if (element) {
                this.modals[key] = new bootstrap.Modal(element);
                
                // Reset form when modal is hidden
                element.addEventListener('hidden.bs.modal', () => {
                    this.resetForm(key);
                });
            }
        });

        // Add weight band functionality
        document.getElementById('addWeightBand')?.addEventListener('click', 
            () => this.addWeightBand());
    }

    setupModalEventListeners() {
        // Form submission handlers
        document.getElementById('serviceProviderForm')?.addEventListener('submit', 
            (e) => this.handleVendorFormSubmit(e));
        
        document.getElementById('supplierForm')?.addEventListener('submit', 
            (e) => this.handleVendorFormSubmit(e));
        
        document.getElementById('routeForm')?.addEventListener('submit', 
            (e) => this.handleRouteFormSubmit(e));
        
        document.getElementById('paymentForm')?.addEventListener('submit', 
            (e) => this.handlePaymentFormSubmit(e));
    }

    showAddVendorModal(type, vendorId = null) {
        this.isEditMode = !!vendorId;
        this.currentEditId = vendorId;
        
        if (type === 'service-provider') {
            this.setupServiceProviderForm(vendorId);
            this.modals.serviceProviderModal?.show();
        } else {
            this.setupSupplierForm(vendorId);
            this.modals.supplierModal?.show();
        }
    }

    setupServiceProviderForm(vendorId = null) {
        const form = document.getElementById('serviceProviderForm');
        const title = document.getElementById('serviceProviderModalTitle');
        const submitBtn = document.getElementById('submitButton');
        
        if (vendorId) {
            // Edit mode - load vendor data
            title.textContent = 'Edit Service Provider';
            submitBtn.textContent = 'Update Service Provider';
            form.action = `/service-providers/${vendorId}`;
            document.getElementById('formMethod').innerHTML = "@method('PUT')";
            
            // Load vendor data via AJAX
            this.loadVendorData(vendorId, 'service_provider');
        } else {
            // Add mode
            title.textContent = 'Add New Service Provider';
            submitBtn.textContent = 'Save Service Provider';
            form.action = '/service-providers';
            document.getElementById('formMethod').innerHTML = '';
        }
    }

    setupSupplierForm(vendorId = null) {
        // Similar setup for supplier form
        const form = document.getElementById('supplierForm');
        const title = document.getElementById('supplierModalTitle');
        const submitBtn = document.getElementById('submitSupplierButton');
        
        if (vendorId) {
            title.textContent = 'Edit Supplier';
            submitBtn.textContent = 'Update Supplier';
            form.action = `/service-providers/${vendorId}`;
            document.getElementById('supplierFormMethod').innerHTML = "@method('PUT')";
            
            this.loadVendorData(vendorId, 'supplier');
        } else {
            title.textContent = 'Add New Supplier';
            submitBtn.textContent = 'Save Supplier';
            form.action = '/service-providers';
            document.getElementById('supplierFormMethod').innerHTML = '';
        }
    }

    async loadVendorData(vendorId, type) {
        try {
            const response = await fetch(`/service-providers/${vendorId}/edit-data`);
            const vendor = await response.json();
            
            if (type === 'service_provider') {
                this.populateServiceProviderForm(vendor);
            } else {
                this.populateSupplierForm(vendor);
            }
        } catch (error) {
            console.error('Error loading vendor data:', error);
            this.suppliersApp.showAlert('Error loading vendor data', 'error');
        }
    }

    populateServiceProviderForm(vendor) {
        document.getElementById('providerName').value = vendor.name || '';
        document.getElementById('vendorCode').value = vendor.vendor_code || '';
        document.getElementById('onboardingDate').value = vendor.onboarding_date || '';
        document.getElementById('openBalance').value = vendor.open_balance || 0;
        document.getElementById('balanceType').value = vendor.balance_type || 'credit';
        document.getElementById('contactPerson').value = vendor.contact_person || '';
        document.getElementById('providerEmail').value = vendor.email || '';
        document.getElementById('providerPhone').value = vendor.phone || '';
        document.getElementById('providerAddress').value = vendor.address || '';
        document.getElementById('accountNumber').value = vendor.account_number || '';
        document.getElementById('providerStatus').value = vendor.status || 'active';
        document.getElementById('providerNotes').value = vendor.notes || '';
        document.getElementById('is_default_provider').checked = vendor.is_default || false;
        
        // Populate shipping methods
        if (vendor.shipping_methods) {
            vendor.shipping_methods.forEach(method => {
                const checkbox = document.querySelector(`input[name="shipping_methods[]"][value="${method}"]`);
                if (checkbox) checkbox.checked = true;
            });
        }
    }

    async handleVendorFormSubmit(event) {
        event.preventDefault();
        const form = event.target;
        const formData = new FormData(form);
        const submitButton = form.querySelector('button[type="submit"]');
        const spinner = submitButton.querySelector('.spinner-border');
        
        // Show loading state
        submitButton.disabled = true;
        spinner.classList.remove('d-none');
        
        // Clear previous errors
        this.clearFormErrors(form);
        
        try {
            const response = await fetch(form.action, {
                method: form.querySelector('input[name="_method"]') ? 'PUT' : 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            });

            const result = await response.json();

            if (response.ok) {
                this.suppliersApp.showAlert(
                    this.isEditMode ? 'Vendor updated successfully!' : 'Vendor created successfully!', 
                    'success'
                );
                this.modals.serviceProviderModal?.hide();
                this.modals.supplierModal?.hide();
                setTimeout(() => window.location.reload(), 1500);
            } else {
                // Display validation errors
                this.displayFormErrors(form, result.errors || {});
                throw new Error(result.message || 'Form submission failed');
            }
        } catch (error) {
            console.error('Error submitting form:', error);
            this.suppliersApp.showAlert(
                error.message || 'Error processing your request', 
                'error'
            );
        } finally {
            // Restore button state
            submitButton.disabled = false;
            spinner.classList.add('d-none');
        }
    }

    displayFormErrors(form, errors) {
        const errorsContainer = form.querySelector('#formErrors');
        if (errorsContainer && Object.keys(errors).length > 0) {
            errorsContainer.innerHTML = '';
            errorsContainer.classList.remove('d-none');
            
            Object.values(errors).forEach(errorMessages => {
                errorMessages.forEach(error => {
                    errorsContainer.innerHTML += `<div>${error}</div>`;
                });
            });
        }
        
        // Highlight invalid fields
        Object.keys(errors).forEach(fieldName => {
            const input = form.querySelector(`[name="${fieldName}"]`);
            if (input) {
                input.classList.add('is-invalid');
            }
        });
    }

    clearFormErrors(form) {
        const errorsContainer = form.querySelector('#formErrors');
        if (errorsContainer) {
            errorsContainer.classList.add('d-none');
            errorsContainer.innerHTML = '';
        }
        
        // Remove invalid styling
        form.querySelectorAll('.is-invalid').forEach(input => {
            input.classList.remove('is-invalid');
        });
    }

    resetForm(modalType) {
        this.isEditMode = false;
        this.currentEditId = null;
        
        const form = document.getElementById(modalType === 'serviceProviderModal' ? 
            'serviceProviderForm' : 'supplierForm');
        if (form) {
            form.reset();
            this.clearFormErrors(form);
        }
    }

    showPaymentModal(vendorId, vendorName) {
        document.getElementById('paymentVendorId').value = vendorId;
        document.getElementById('paymentVendorName').value = vendorName;
        if (this.modals.paymentModal) {
            this.modals.paymentModal.show();
        }
    }

    showRouteModal(vendorId = null) {
        if (vendorId) {
            document.getElementById('routeVendorId').value = vendorId;
        }
        if (this.modals.routeModal) {
            this.modals.routeModal.show();
        }
    }

    addWeightBand() {
        const table = document.getElementById('weightBandsTable');
        if (!table) return;

        const rowCount = table.rows.length;
        const newRow = table.insertRow();
        
        newRow.innerHTML = `
            <td><input type="number" class="form-control form-control-sm" name="weight_bands[${rowCount}][min]" step="0.01"></td>
            <td><input type="number" class="form-control form-control-sm" name="weight_bands[${rowCount}][max]" step="0.01"></td>
            <td>
                <input type="number" class="form-control form-control-sm" name="weight_bands[${rowCount}][rate]" step="0.01">
                <button type="button" class="btn btn-sm btn-danger mt-1" onclick="this.closest('tr').remove()">
                    <i class="ri-delete-bin-line"></i>
                </button>
            </td>
        `;
    }

    async handleRouteFormSubmit(event) {
        event.preventDefault();
        // Implement route form submission
        this.suppliersApp.showAlert('Route functionality coming soon!', 'info');
    }

    async handlePaymentFormSubmit(event) {
        event.preventDefault();
        // Implement payment form submission
        this.suppliersApp.showAlert('Payment functionality coming soon!', 'info');
    }
}