class PackageManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
        this.packageCount = 0;
        this.totalWeight = 0;
    }

    init() {
        this.setupEventListeners();
    }

    setupEventListeners() {
        const numPiecesInput = document.getElementById('numPieces');
        if (numPiecesInput) {
            numPiecesInput.addEventListener('change', () => this.generatePackageForms());
        }
    }

    generatePackageForms() {
        const numPieces = parseInt(document.getElementById('numPieces').value);
        const container = document.getElementById('packageFormsContainer');
        const summary = document.getElementById('packageSummary');
        
        if (!container) return;
        
        container.innerHTML = '';
        this.packageCount = numPieces;
        this.totalWeight = 0;
        
        if (numPieces > 0) {
            for (let i = 1; i <= numPieces; i++) {
                const packageHtml = this.createPackageHtml(i);
                container.innerHTML += packageHtml;
            }
            
            if (summary) summary.classList.remove('d-none');
            this.updatePackageSummary();
            this.setupPackageEventListeners();
        } else {
            if (summary) summary.classList.add('d-none');
        }
    }

    createPackageHtml(index) {
        return `
            <div class="accordion package-accordion" id="packageAccordion${index}">
                <div class="accordion-item">
                    <h2 class="accordion-header">
                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#packageCollapse${index}">
                            Package ${index}
                        </button>
                    </h2>
                    <div id="packageCollapse${index}" class="accordion-collapse collapse">
                        <div class="accordion-body">
                            <div class="row">
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Length (cm)</label>
                                    <input type="number" class="form-control package-dimension" data-package="${index}" data-dimension="length" name="packages[${index}][length]">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Width (cm)</label>
                                    <input type="number" class="form-control package-dimension" data-package="${index}" data-dimension="width" name="packages[${index}][width]">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Height (cm)</label>
                                    <input type="number" class="form-control package-dimension" data-package="${index}" data-dimension="height" name="packages[${index}][height]">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Weight (kg)</label>
                                    <input type="number" class="form-control package-weight" data-package="${index}" name="packages[${index}][weight]" step="0.1">
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Contents Description</label>
                                <textarea class="form-control" name="packages[${index}][contents]" rows="3" placeholder="Describe the contents of this package"></textarea>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }

    setupPackageEventListeners() {
        document.querySelectorAll('.package-weight').forEach(input => {
            input.addEventListener('change', () => this.updatePackageSummary());
        });
    }

    updatePackageSummary() {
        let totalWeight = 0;
        
        document.querySelectorAll('.package-weight').forEach(input => {
            if (input.value) {
                totalWeight += parseFloat(input.value);
            }
        });
        
        const totalPackagesCount = document.getElementById('totalPackagesCount');
        const totalWeightDisplay = document.getElementById('totalWeightDisplay');
        const summaryPackages = document.getElementById('summaryPackages');
        const summaryWeight = document.getElementById('summaryWeight');
        
        if (totalPackagesCount) totalPackagesCount.textContent = this.packageCount;
        if (totalWeightDisplay) totalWeightDisplay.textContent = totalWeight.toFixed(2);
        if (summaryPackages) summaryPackages.textContent = this.packageCount;
        if (summaryWeight) summaryWeight.textContent = totalWeight.toFixed(2);
        
        this.totalWeight = totalWeight;
        
        // Trigger rate calculation if route manager is available
        if (window.shipmentForm && window.shipmentForm.getModule('routeManager')) {
            window.shipmentForm.getModule('routeManager').calculateRouteRates();
        }
    }

    validateSection() {
        const stepManager = this.shipmentForm.getModule('stepManager');
        if (stepManager && stepManager.validateStep(2)) {
            stepManager.moveToStep(3);
        }
    }

    getPackageCount() {
        return this.packageCount;
    }

    getTotalWeight() {
        return this.totalWeight;
    }
}