class CustomerManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
    }

    init() {
        this.setupEventListeners();
    }

    setupEventListeners() {
        const customerSelect = document.getElementById('customer_id');
        if (customerSelect) {
            customerSelect.addEventListener('change', () => this.loadCustomerDetails());
        }
        
        const sendingCheckbox = document.getElementById('sendingOnBehalf');
        if (sendingCheckbox) {
            sendingCheckbox.addEventListener('change', () => this.toggleSenderInfo());
        }
    }

    loadCustomerDetails() {
        const customerId = document.getElementById('customer_id').value;
        const selectedOption = document.getElementById('customer_id').selectedOptions[0];
        
        if (customerId && selectedOption) {
            const phone = selectedOption.getAttribute('data-phone');
            const email = selectedOption.getAttribute('data-email');
            const name = selectedOption.textContent.split(' - ')[0];
            
            this.displayCustomerDetails(name, phone, email);
            this.updateVerificationFields(phone);
        } else {
            this.hideCustomerDetails();
        }
    }

    displayCustomerDetails(name, phone, email) {
        const nameDisplay = document.getElementById('customerNameDisplay');
        const phoneDisplay = document.getElementById('customerPhoneDisplay');
        const emailDisplay = document.getElementById('customerEmailDisplay');
        const customerDetails = document.getElementById('customerDetails');
        
        if (nameDisplay) nameDisplay.textContent = name;
        if (phoneDisplay) phoneDisplay.textContent = phone;
        if (emailDisplay) emailDisplay.textContent = email;
        if (customerDetails) customerDetails.classList.remove('d-none');
    }

    hideCustomerDetails() {
        const customerDetails = document.getElementById('customerDetails');
        if (customerDetails) customerDetails.classList.add('d-none');
    }

    updateVerificationFields(phone) {
        const otpPhone = document.getElementById('otp_phone');
        const whatsappNumber = document.getElementById('whatsapp_number');
        
        if (otpPhone) otpPhone.value = phone;
        if (whatsappNumber) whatsappNumber.value = phone;
    }

    toggleSenderInfo() {
        const senderSection = document.getElementById('senderInfoSection');
        const sendingCheckbox = document.getElementById('sendingOnBehalf');
        
        if (senderSection && sendingCheckbox) {
            if (sendingCheckbox.checked) {
                senderSection.classList.remove('d-none');
            } else {
                senderSection.classList.add('d-none');
            }
        }
    }

    validateSection() {
        const stepManager = this.shipmentForm.getModule('stepManager');
        if (stepManager && stepManager.validateStep(1)) {
            stepManager.moveToStep(2);
        }
    }

    async saveNewCustomer() {
        const formData = {
            name: document.getElementById('new_customer_name').value,
            phone: document.getElementById('new_customer_phone').value,
            id_number: document.getElementById('new_customer_id').value,
            email: document.getElementById('new_customer_email').value,
            address: document.getElementById('new_customer_address').value,
            city: document.getElementById('new_customer_city').value,
            country_id: document.getElementById('new_customer_country').value
        };

        try {
            const response = await fetch('/api/customers', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCsrfToken()
                },
                body: JSON.stringify(formData)
            });

            const result = await response.json();

            if (result.success) {
                this.addCustomerToDropdown(result.customer);
                this.showAlert('Customer created successfully!', 'success');
                
                // Close modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('newCustomerModal'));
                if (modal) modal.hide();
            } else {
                throw new Error(result.message || 'Failed to create customer');
            }
        } catch (error) {
            this.showAlert('Error creating customer: ' + error.message, 'error');
        }
    }

    addCustomerToDropdown(customer) {
        const select = document.getElementById('customer_id');
        const option = new Option(`${customer.name} - ${customer.phone}`, customer.id);
        option.setAttribute('data-phone', customer.phone);
        option.setAttribute('data-email', customer.email || '');
        select.add(option);
        select.value = customer.id;
        
        this.loadCustomerDetails();
    }

    getCsrfToken() {
        const csrfMeta = document.querySelector('meta[name="csrf-token"]');
        return csrfMeta ? csrfMeta.getAttribute('content') : '';
    }

    showAlert(message, type = 'info') {
        if (window.shipmentForm) {
            window.shipmentForm.showAlert(message, type);
        } else {
            alert(message);
        }
    }
}