<?php

namespace App\Services;

use App\Models\{Shipment, Customer, Package, Service};
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ShipmentService
{
    public function __construct(
        protected PackageService $packageService,
        protected FreightCalculator $freightCalculator,
        protected LabelService $labelService
    ) {}

    public function createShipment($request): Shipment
    {
        return DB::transaction(function () use ($request) {
            $customer = Customer::firstOrCreate(
                ['phone' => $request->customer_phone],
                [
                    'name' => $request->customer_name,
                    'email' => $request->customer_email,
                    'password' => bcrypt(Str::random(8)),
                    'address' => $request->customer_address,
                    'national_id' => $request->customer_national_id,
                ]
            );

            $shipment = Shipment::create([
                'customer_id' => $customer->id,
                'mode' => $request->mode,
                'category_id' => $request->category_id,
                'origin_country_id' => $request->origin_country_id,
                'destination_country_id' => $request->destination_country_id,
                'status' => 'pending'
            ]);

            $freightTotal = 0;
            foreach ($request->packages as $i => $pkg) {
                $result = $this->packageService->processPackage($pkg, $i, $shipment, $request);
                $freightTotal += $result['freight_cost'];
            }

            if ($request->services) {
                foreach ($request->services as $srv) {
                    $rate = convertCurrency($srv['charge'], $srv['currency'], 'KES');
                    $shipment->services()->attach($srv['service_id'], [
                        'charge' => $rate,
                        'currency' => 'KES',
                    ]);
                }
            }

            $shipment->update(['freight' => $freightTotal]);

            return $shipment;
        });
    }
}
