<?php

namespace App\Services;

use App\Models\ShipmentRoute;
use Illuminate\Support\Collection;

class ChargeableWeightService
{
    /**
     * Calculate the chargeable weight for a package based on the route.
     *
     * @param array $package ['weight', 'length', 'width', 'height']
     * @param ShipmentRoute $route
     * @return float
     */
    public function calculate(array $package, ShipmentRoute $route): float
    {
        $weight = floatval($package['weight'] ?? 0);
        $length = floatval($package['length'] ?? 0);
        $width  = floatval($package['width'] ?? 0);
        $height = floatval($package['height'] ?? 0);

        $divisor = $route->shipmentMode->volume_divisor ?? 5000; // Default air divisor

        $volumetricWeight = ($length * $width * $height) / $divisor;

        // Unit type can be kg, cbm, cuft, etc.
        $unit = strtolower($route->unit_type);

        switch ($unit) {
            case 'cbm':
                return ($length * $width * $height) / 1000000; // dimensions in cm → m³
            case 'cuft':
                return ($length * $width * $height) / 28316.8466; // cm³ → ft³
            case 'kg':
                return $this->resolveByBasis($route->weight_basis, $weight, $volumetricWeight);
            default:
                return $this->resolveByBasis($route->weight_basis, $weight, $volumetricWeight);
        }
    }

    /**
     * Apply the weight basis logic
     *
     * @param string|null $basis
     * @param float $gross
     * @param float $volumetric
     * @return float
     */
    private function resolveByBasis(?string $basis, float $gross, float $volumetric): float
    {
        return match ($basis) {
            'gross'      => $gross,
            'volumetric' => $volumetric,
            'higher'     => max($gross, $volumetric),
            default      => $gross, // Fallback
        };
    }
}
