<?php

namespace App\Http\Controllers\Vendors;

use App\Http\Controllers\Controller;
use App\Models\Vendors\ServiceProvider; // Correct namespace
use App\Models\Vendors\ServiceProviderRoute; // Add Vendors namespace
use App\Models\Vendors\ServiceProviderWeightBand; // Add Vendors namespace
use App\Models\Vendors\SupplierPayment; // Add Vendors namespace
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ServiceProviderController extends Controller
{
    /**
     * Display a listing of service providers and suppliers.
     */
    public function index()
    {
        try {
            $serviceProviders = ServiceProvider::where('type', 'service_provider')
                ->withCount('routes')
                ->with('routes.weightBands')
                ->orderBy('is_default', 'desc')
                ->orderBy('name')
                ->paginate(10, ['*'], 'providers_page');

            $suppliers = ServiceProvider::where('type', 'supplier')
                ->withCount('payments')
                ->orderBy('name')
                ->paginate(10, ['*'], 'suppliers_page');

            return view('vendor.service-providers.index', compact('serviceProviders', 'suppliers'));
        } catch (\Exception $e) {
            return redirect()->route('dashboard')->with('error', 'Error loading vendors: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for creating a new service provider or supplier.
     */
    public function create($type = 'service_provider')
    {
        if (!in_array($type, ['service_provider', 'supplier'])) {
            abort(404);
        }

        return view('vendor.service-providers.create', compact('type'));
    }

    /**
     * Store a newly created service provider or supplier.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:service_provider,supplier',
            'onboarding_date' => 'required|date',
            'open_balance' => 'required|numeric|min:0',
            'balance_type' => 'required|in:credit,debit',
            'contact_person' => 'nullable|string|max:255',
            'email' => 'nullable|email|unique:service_providers,email',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'account_number' => 'nullable|string|max:50',
            'shipping_methods' => 'nullable|array',
            'shipping_methods.*' => 'in:air,sea,land,rail',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $vendorCode = $this->generateVendorCode($validated['type']);

            $serviceProvider = ServiceProvider::create([
                'name' => $validated['name'],
                'type' => $validated['type'],
                'vendor_code' => $vendorCode,
                'onboarding_date' => $validated['onboarding_date'],
                'open_balance' => $validated['open_balance'],
                'balance_type' => $validated['balance_type'],
                'contact_person' => $validated['contact_person'],
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'address' => $validated['address'],
                'account_number' => $validated['account_number'],
                'status' => 'active',
                'is_default' => $request->has('is_default'),
                'notes' => $validated['notes'],
            ]);

            // If it's a default service provider, create a default route
            if ($serviceProvider->is_default && $serviceProvider->type === 'service_provider') {
                $this->createDefaultRoute($serviceProvider);
            }

            DB::commit();

            return redirect()->route('service-providers.index')
                ->with('success', ucfirst(str_replace('_', ' ', $validated['type'])) . ' created successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', 'Error creating vendor: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified service provider or supplier.
     */
    public function show(ServiceProvider $serviceProvider)
    {
        $serviceProvider->load(['routes.weightBands', 'payments' => function($query) {
            $query->orderBy('payment_date', 'desc')->limit(10);
        }]);

        $tab = request()->get('tab', 'details');

        return view('vendor.service-providers.show', compact('serviceProvider', 'tab'));
    }

    /**
     * Show the form for editing the specified service provider or supplier.
     */
    public function edit(ServiceProvider $serviceProvider)
    {
        return view('vendor.service-providers.edit', compact('serviceProvider'));
    }

    /**
     * Update the specified service provider or supplier.
     */
    public function update(Request $request, ServiceProvider $serviceProvider)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'onboarding_date' => 'required|date',
            'open_balance' => 'required|numeric|min:0',
            'balance_type' => 'required|in:credit,debit',
            'contact_person' => 'nullable|string|max:255',
            'email' => 'nullable|email|unique:service_providers,email,' . $serviceProvider->id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'account_number' => 'nullable|string|max:50',
            'status' => 'required|in:active,inactive',
            'notes' => 'nullable|string',
        ]);

        try {
            $serviceProvider->update([
                'name' => $validated['name'],
                'onboarding_date' => $validated['onboarding_date'],
                'open_balance' => $validated['open_balance'],
                'balance_type' => $validated['balance_type'],
                'contact_person' => $validated['contact_person'],
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'address' => $validated['address'],
                'account_number' => $validated['account_number'],
                'status' => $validated['status'],
                'is_default' => $request->has('is_default'),
                'notes' => $validated['notes'],
            ]);

            return redirect()->route('service-providers.index')
                ->with('success', ucfirst(str_replace('_', ' ', $serviceProvider->type)) . ' updated successfully!');

        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Error updating vendor: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified service provider or supplier.
     */
    public function destroy(ServiceProvider $serviceProvider)
    {
        DB::beginTransaction();
        try {
            // Check if service provider has associated records
            if ($serviceProvider->routes()->exists()) {
                return redirect()->back()->with('error', 'Cannot delete vendor with associated routes.');
            }

            if ($serviceProvider->payments()->exists()) {
                return redirect()->back()->with('error', 'Cannot delete vendor with associated payments.');
            }

            $serviceProvider->delete();
            DB::commit();

            return redirect()->route('service-providers.index')
                ->with('success', ucfirst(str_replace('_', ' ', $serviceProvider->type)) . ' deleted successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error deleting vendor: ' . $e->getMessage());
        }
    }

    /**
     * Toggle vendor status (active/inactive)
     */
    public function toggleStatus(ServiceProvider $serviceProvider)
    {
        try {
            $serviceProvider->update([
                'status' => $serviceProvider->status === 'active' ? 'inactive' : 'active'
            ]);

            return response()->json([
                'success' => true,
                'status' => $serviceProvider->status,
                'message' => 'Status updated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Route Management Methods
     */
    public function storeRoute(Request $request, ServiceProvider $serviceProvider)
    {
        $validated = $request->validate([
            'route_name' => 'required|string|max:255',
            'origin' => 'required|string|max:255',
            'destination' => 'required|string|max:255',
            'minimum_chargeable' => 'required|numeric|min:0',
            'volumetric_divisor' => 'required|numeric|min:0',
            'chargeable_type' => 'required|in:volumetric_weight,gross_weight,net_volume',
            'shipping_method' => 'required|in:air,sea,land,rail',
            'transit_days' => 'nullable|integer|min:1',
            'rate_per_kg' => 'nullable|numeric|min:0',
            'rate_per_volume' => 'nullable|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            $route = $serviceProvider->routes()->create($validated);

            // Create default weight bands
            $defaultBands = [
                ['min_weight' => 0, 'max_weight' => 25, 'rate' => $validated['rate_per_kg'] ?? 0],
                ['min_weight' => 25, 'max_weight' => 50, 'rate' => $validated['rate_per_kg'] ?? 0],
                ['min_weight' => 50, 'max_weight' => 100, 'rate' => $validated['rate_per_kg'] ?? 0],
                ['min_weight' => 100, 'max_weight' => 500, 'rate' => $validated['rate_per_kg'] ?? 0],
                ['min_weight' => 500, 'max_weight' => 1000, 'rate' => $validated['rate_per_kg'] ?? 0],
            ];

            foreach ($defaultBands as $band) {
                if ($band['rate'] > 0) {
                    $route->weightBands()->create($band);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'route' => $route->load('weightBands'),
                'message' => 'Route created successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error creating route: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateRoute(Request $request, ServiceProviderRoute $route)
    {
        $validated = $request->validate([
            'route_name' => 'required|string|max:255',
            'origin' => 'required|string|max:255',
            'destination' => 'required|string|max:255',
            'minimum_chargeable' => 'required|numeric|min:0',
            'volumetric_divisor' => 'required|numeric|min:0',
            'chargeable_type' => 'required|in:volumetric_weight,gross_weight,net_volume',
            'shipping_method' => 'required|in:air,sea,land,rail',
            'transit_days' => 'nullable|integer|min:1',
            'rate_per_kg' => 'nullable|numeric|min:0',
            'rate_per_volume' => 'nullable|numeric|min:0',
        ]);

        try {
            $route->update($validated);

            return response()->json([
                'success' => true,
                'route' => $route,
                'message' => 'Route updated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating route: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroyRoute(ServiceProviderRoute $route)
    {
        try {
            $route->delete();

            return response()->json([
                'success' => true,
                'message' => 'Route deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting route: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Weight Band Management Methods
     */
    public function storeWeightBand(Request $request, ServiceProviderRoute $route)
    {
        $validated = $request->validate([
            'min_weight' => 'required|numeric|min:0',
            'max_weight' => 'required|numeric|min:0|gt:min_weight',
            'rate' => 'required|numeric|min:0',
        ]);

        try {
            $weightBand = $route->weightBands()->create($validated);

            return response()->json([
                'success' => true,
                'weightBand' => $weightBand,
                'message' => 'Weight band created successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating weight band: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateWeightBand(Request $request, ServiceProviderWeightBand $weightBand)
    {
        $validated = $request->validate([
            'min_weight' => 'required|numeric|min:0',
            'max_weight' => 'required|numeric|min:0|gt:min_weight',
            'rate' => 'required|numeric|min:0',
        ]);

        try {
            $weightBand->update($validated);

            return response()->json([
                'success' => true,
                'weightBand' => $weightBand,
                'message' => 'Weight band updated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating weight band: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroyWeightBand(ServiceProviderWeightBand $weightBand)
    {
        try {
            $weightBand->delete();

            return response()->json([
                'success' => true,
                'message' => 'Weight band deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting weight band: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Payment Management Methods
     */
    public function paymentIndex(ServiceProvider $serviceProvider)
    {
        $payments = $serviceProvider->payments()
            ->with('createdBy')
            ->orderBy('payment_date', 'desc')
            ->paginate(10);

        return view('vendor.service-providers.payments.index', compact('serviceProvider', 'payments'));
    }

    public function paymentCreate(ServiceProvider $serviceProvider)
    {
        return view('vendor.service-providers.payments.create', compact('serviceProvider'));
    }

    public function paymentStore(Request $request, ServiceProvider $serviceProvider)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:bank_transfer,cash,cheque,online',
            'reference_number' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            $payment = $serviceProvider->payments()->create([
                'amount' => $validated['amount'],
                'payment_date' => $validated['payment_date'],
                'payment_method' => $validated['payment_method'],
                'reference_number' => $validated['reference_number'],
                'notes' => $validated['notes'],
                'created_by' => auth()->id(),
            ]);

            // Update service provider balance
            $serviceProvider->update([
                'open_balance' => $serviceProvider->open_balance - $validated['amount']
            ]);

            return redirect()->route('service-providers.payments.index', $serviceProvider)
                ->with('success', 'Payment recorded successfully!');

        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Error recording payment: ' . $e->getMessage());
        }
    }

    /**
     * Summary and Reports Methods
     */
    public function summary(ServiceProvider $serviceProvider)
    {
        $summary = $this->generateSummaryData($serviceProvider);
        $tab = request()->get('tab', 'overview');

        return view('vendor.service-providers.summary', compact('serviceProvider', 'summary', 'tab'));
    }

    public function generateSummary(ServiceProvider $serviceProvider)
    {
        $summary = $this->generateSummaryData($serviceProvider);

        return response()->json([
            'success' => true,
            'summary' => $summary
        ]);
    }

    /**
     * API Methods
     */
    public function search(Request $request)
    {
        $query = $request->get('q');
        $type = $request->get('type', 'all');

        $providers = ServiceProvider::when($type !== 'all', function($q) use ($type) {
                $q->where('type', $type);
            })
            ->where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('vendor_code', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            })
            ->where('status', 'active')
            ->get(['id', 'name', 'vendor_code', 'type', 'email']);

        return response()->json($providers);
    }

    public function getRoutes(ServiceProvider $serviceProvider)
    {
        $routes = $serviceProvider->routes()->with('weightBands')->get();

        return response()->json([
            'success' => true,
            'routes' => $routes
        ]);
    }

    public function calculateRates(Request $request)
    {
        $validated = $request->validate([
            'route_id' => 'required|exists:service_provider_routes,id',
            'weight' => 'required|numeric|min:0',
            'volume' => 'nullable|numeric|min:0',
        ]);

        try {
            $route = ServiceProviderRoute::with('weightBands')->find($validated['route_id']);
            $rates = $this->calculateShippingRates($route, $validated['weight'], $validated['volume']);

            return response()->json([
                'success' => true,
                'rates' => $rates
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error calculating rates: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper Methods
     */
    private function generateVendorCode($type)
    {
        $prefix = $type === 'service_provider' ? 'SP' : 'SUP';
        $latest = ServiceProvider::where('type', $type)->latest()->first();
        $number = $latest ? intval(substr($latest->vendor_code, 3)) + 1 : 1;

        return $prefix . '-' . str_pad($number, 3, '0', STR_PAD_LEFT);
    }

    private function createDefaultRoute(ServiceProvider $serviceProvider)
    {
        $route = $serviceProvider->routes()->create([
            'route_name' => 'Default Inhouse Route',
            'origin' => 'Nairobi',
            'destination' => 'Nairobi',
            'minimum_chargeable' => 0,
            'volumetric_divisor' => 5000,
            'chargeable_type' => 'volumetric_weight',
            'shipping_method' => 'land',
            'transit_days' => 1,
            'rate_per_kg' => 50,
            'is_active' => true,
        ]);

        // Create default weight bands for inhouse route
        $defaultBands = [
            ['min_weight' => 0, 'max_weight' => 25, 'rate' => 50],
            ['min_weight' => 25, 'max_weight' => 50, 'rate' => 45],
            ['min_weight' => 50, 'max_weight' => 100, 'rate' => 40],
            ['min_weight' => 100, 'max_weight' => 500, 'rate' => 35],
            ['min_weight' => 500, 'max_weight' => 1000, 'rate' => 30],
        ];

        foreach ($defaultBands as $band) {
            $route->weightBands()->create($band);
        }
    }

    private function generateSummaryData(ServiceProvider $serviceProvider)
    {
        $startDate = request()->get('start_date', now()->subMonth());
        $endDate = request()->get('end_date', now());

        return [
            'total_shipments' => 0, // You'll need to implement this based on your shipment model
            'total_revenue' => 0,
            'total_cost' => 0,
            'profit_margin' => 0,
            'payment_summary' => [
                'total_paid' => $serviceProvider->payments()->sum('amount'),
                'last_payment_date' => $serviceProvider->payments()->max('payment_date'),
                'outstanding_balance' => $serviceProvider->open_balance,
            ],
            'period' => [
                'start_date' => $startDate,
                'end_date' => $endDate,
            ]
        ];
    }

    private function calculateShippingRates($route, $weight, $volume = null)
    {
        $rates = [
            'base_rate' => 0,
            'weight_charge' => 0,
            'volume_charge' => 0,
            'total_charge' => 0,
            'applicable_band' => null,
        ];

        // Find applicable weight band
        $applicableBand = $route->weightBands()
            ->where('min_weight', '<=', $weight)
            ->where('max_weight', '>=', $weight)
            ->first();

        if ($applicableBand) {
            $rates['weight_charge'] = $weight * $applicableBand->rate;
            $rates['applicable_band'] = $applicableBand;
        }

        // Calculate volume charge if volume is provided
        if ($volume && $route->rate_per_volume) {
            $rates['volume_charge'] = $volume * $route->rate_per_volume;
        }

        $rates['total_charge'] = max(
            $route->minimum_chargeable,
            $rates['weight_charge'] + $rates['volume_charge']
        );

        return $rates;
    }
}