<?php

namespace App\Http\Controllers;

use App\Models\ShipmentRoute;
use App\Models\Country;
use App\Models\Currency;
use App\Models\ShipmentMode;
use App\Models\CargoCategory;
use Illuminate\Http\Request;

class ShipmentRouteController extends Controller
{
    public function index()
    {
        $routes = ShipmentRoute::with(['originCountry', 'destinationCountry', 'shipmentMode', 'cargoCategory', 'currency'])
            ->latest()
            ->paginate(10);

        return view('shipment_routes.index', compact('routes'));
    }

    public function create()
    {
        return view('shipment_routes.create', [
            'countries' => Country::all(),
            'currencies' => Currency::all(),
            'shipmentModes' => ShipmentMode::with('measure')->get(),
            'cargoCategories' => CargoCategory::all(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'origin_country_id' => 'required|exists:countries,id',
            'destination_country_id' => 'required|exists:countries,id',
            'shipment_mode_id' => 'required|exists:shipment_modes,id',
            'cargo_category_id' => 'required|exists:cargo_categories,id',
            'currency_id' => 'required|exists:currencies,id',
            'rate_per_unit' => 'nullable|numeric',
            'flat_rate' => 'nullable|numeric',
            'minimum_weight' => 'required|numeric|min:0',
            'unit_type' => 'required|string|in:kg,CBM,CUFT',
            'weight_basis' => 'required|in:gross,volumetric,higher',
        ]);

        ShipmentRoute::create($validated);

        return redirect()->route('shipment-routes.index')
            ->with('success', 'Shipment route created successfully.');
    }

    public function edit(ShipmentRoute $shipment_route)
    {
        return view('shipment_routes.edit', [
            'route' => $shipment_route,
            'countries' => Country::all(),
            'currencies' => Currency::all(),
            'shipmentModes' => ShipmentMode::with('measure')->get(),
            'cargoCategories' => CargoCategory::all(),
        ]);
    }

    public function update(Request $request, ShipmentRoute $shipment_route)
    {
        $validated = $request->validate([
            'origin_country_id' => 'required|exists:countries,id',
            'destination_country_id' => 'required|exists:countries,id',
            'shipment_mode_id' => 'required|exists:shipment_modes,id',
            'cargo_category_id' => 'required|exists:cargo_categories,id',
            'currency_id' => 'required|exists:currencies,id',
            'rate_per_unit' => 'nullable|numeric',
            'flat_rate' => 'nullable|numeric',
            'minimum_weight' => 'required|numeric|min:0',
            'unit_type' => 'required|string|in:kg,CBM,CUFT',
            'weight_basis' => 'required|in:gross,volumetric,higher',
        ]);

        $shipment_route->update($validated);

        return redirect()->route('shipment-routes.index')
            ->with('success', 'Shipment route updated successfully.');
    }

    public function destroy(ShipmentRoute $shipment_route)
    {
        $shipment_route->delete();

        return redirect()->route('shipment-routes.index')
            ->with('success', 'Shipment route deleted successfully.');
    }

    /**
     * API: Find shipment route details
     */
public function findRoute(Request $request)
{
    \Log::info('Find Route Request:', $request->all()); // Log the incoming request

    try {
        $validated = $request->validate([
            'origin_country_id' => 'required|integer|exists:countries,id',
            'destination_country_id' => 'required|integer|exists:countries,id',
            'mode' => 'required|string', // Changed from exists:shipment_modes,code to make debugging easier
            'category_id' => 'required|integer|exists:cargo_categories,id',
        ]);

        \Log::info('Validated Data:', $validated); // Log validated data

        $shipmentMode = ShipmentMode::where('code', $validated['mode'])->first();
        
        if (!$shipmentMode) {
            \Log::warning('Shipment mode not found', ['mode' => $validated['mode']]);
            return response()->json([
                'success' => false,
                'message' => 'Shipment mode not found',
                'volumetric_divisor' => 6000
            ], 200);
        }

        $route = ShipmentRoute::with('currency')
            ->where('origin_country_id', $validated['origin_country_id'])
            ->where('destination_country_id', $validated['destination_country_id'])
            ->where('shipment_mode_id', $shipmentMode->id)
            ->where('cargo_category_id', $validated['category_id'])
            ->latest()
            ->first();

        if (!$route) {
            \Log::info('No route found for criteria', $validated);
            return response()->json([
                'success' => false,
                'message' => 'No route found for the selected criteria',
                'volumetric_divisor' => 6000
            ], 200);
        }

        return response()->json([
            'success' => true,
            'rate_per_unit' => $route->rate_per_unit,
            'flat_rate' => $route->flat_rate,
            'unit_type' => $route->unit_type,
            'weight_basis' => $route->weight_basis,
            'minimum_weight' => $route->minimum_weight,
            'currency' => $route->currency->code,
            'volumetric_divisor' => $route->volumetric_divisor ?? 6000,
            'cbm_divisor' => $route->cbm_divisor ?? 1000000,
            'cuft_divisor' => $route->cuft_divisor ?? 28316.8466,
        ]);

    } catch (\Exception $e) {
        \Log::error('Find Route Error:', ['error' => $e->getMessage(), 'trace' => $e->getTrace()]);
        return response()->json([
            'success' => false,
            'message' => $e->getMessage(),
            'volumetric_divisor' => 6000
        ], 200);
    }
}

    /**
     * API: Returns breakdown of all available rates for a given route
     */
    public function breakdown(Request $request)
    {
        $validated = $request->validate([
            'origin' => 'required|integer|exists:countries,id',
            'dest' => 'required|integer|exists:countries,id',
            'category' => 'required|integer|exists:cargo_categories,id',
            'mode' => 'required|string|exists:shipment_modes,code',
        ]);

        $mode = ShipmentMode::where('code', $validated['mode'])->first();

        $breakdown = ShipmentRoute::where([
                'origin_country_id' => $validated['origin'],
                'destination_country_id' => $validated['dest'],
                'cargo_category_id' => $validated['category'],
                'shipment_mode_id' => $mode->id,
            ])
            ->orderBy('minimum_weight')
            ->get();

        return response()->json($breakdown);
    }

    /**
     * API: Returns rate change history for a given route
     */
    public function history(Request $request)
    {
        $validated = $request->validate([
            'origin' => 'required|integer|exists:countries,id',
            'dest' => 'required|integer|exists:countries,id',
            'category' => 'required|integer|exists:cargo_categories,id',
            'mode' => 'required|string|exists:shipment_modes,code',
        ]);

        $mode = ShipmentMode::where('code', $validated['mode'])->first();

        $history = ShipmentRoute::with(['user:id,name', 'currency'])
            ->where([
                'origin_country_id' => $validated['origin'],
                'destination_country_id' => $validated['dest'],
                'cargo_category_id' => $validated['category'],
                'shipment_mode_id' => $mode->id,
            ])
            ->orderByDesc('created_at')
            ->get()
            ->map(function ($route) {
                return [
                    'created_at' => $route->created_at->toDateTimeString(),
                    'rate' => $route->rate_per_unit ?? $route->flat_rate,
                    'currency' => $route->currency->code,
                    'user_name' => $route->user->name ?? 'System',
                ];
            });

        return response()->json($history);
    }
}