<?php

namespace App\Http\Controllers;

use App\Models\{
    Customer,
    Country,
    Service,
    Currency,
    Shipment,
    Package,
    ShipmentService,
    ShipmentRoute,
    ShipmentRouteHistory,
    CargoCategory,
    ShipmentMode
};
use App\Http\Requests\{
    StoreShipmentRequest,
    UpdateShipmentRequest,
    FindRouteRequest
};
use App\Services\{
    ShipmentService as ShipmentServiceHandler,
    PackageService,
    LabelService
};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\{DB, Cache, Storage};
use Illuminate\Support\Str;
use PDF;

class ShipmentController extends Controller
{
    protected $shipmentService;
    protected $packageService;
    protected $labelService;

    public function __construct(
        ShipmentServiceHandler $shipmentService,
        PackageService $packageService,
        LabelService $labelService
    ) {
        $this->shipmentService = $shipmentService;
        $this->packageService = $packageService;
        $this->labelService = $labelService;
    }

    public function index()
    {
        $shipments = Shipment::with(['customer', 'packages', 'services.service'])
            ->latest()
            ->paginate(15);

        return view('shipments.index', compact('shipments'));
    }

    public function create()
    {
        $currencyRates = Currency::select('code', 'buying_exchange_rate', 'selling_exchange_rate')
            ->get()
            ->keyBy('code')
            ->map(function($currency) {
                return [
                    'buy' => $currency->buying_exchange_rate,
                    'sell' => $currency->selling_exchange_rate
                ];
            })
            ->toArray();

        return view('shipments.create', [
            'customers' => Customer::orderBy('name')->get(),
            'countries' => Country::orderBy('name')->get(),
            'services' => Service::orderBy('name')->get(),
            'currencies' => Currency::orderBy('code')->get(),
            'cargoCategories' => CargoCategory::orderBy('name')->get(),
            'shipmentModes' => ShipmentMode::orderBy('name')->get(),
            'method' => 'POST',
            'action' => route('shipments.store'),
            'shipment' => null,
            'currencyRates' => $currencyRates
        ]);
    }

    public function store(StoreShipmentRequest $request)
    {
        try {
            DB::transaction(function () use ($request) {
                $shipment = $this->shipmentService->createShipment($request->validated());
                
                $this->packageService->processPackages(
                    $request->packages,
                    $shipment,
                    $request->origin_country_id,
                    $request->destination_country_id,
                    $request->mode
                );
                
                if ($request->services) {
                    $this->shipmentService->addServices($shipment, $request->services);
                }
                
                $this->shipmentService->updateTotals($shipment);
            });

            return redirect()->route('shipments.index')
                ->with('success', 'Shipment created successfully.');
        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'Error creating shipment: ' . $e->getMessage());
        }
    }

    public function show(Shipment $shipment)
    {
        $shipment->load([
            'customer', 
            'packages', 
            'services.service', 
            'chargeCurrency',
            'originCountry',
            'destinationCountry'
        ]);

        return view('shipments.show', compact('shipment'));
    }

    public function edit(Shipment $shipment)
    {
        $currencyRates = Currency::select('code', 'buying_exchange_rate', 'selling_exchange_rate')
            ->get()
            ->keyBy('code')
            ->map(function($currency) {
                return [
                    'buy' => $currency->buying_exchange_rate,
                    'sell' => $currency->selling_exchange_rate
                ];
            })
            ->toArray();

        return view('shipments.create', [
            'customers' => Customer::orderBy('name')->get(),
            'countries' => Country::orderBy('name')->get(),
            'services' => Service::orderBy('name')->get(),
            'currencies' => Currency::orderBy('code')->get(),
            'cargoCategories' => CargoCategory::orderBy('name')->get(),
            'shipmentModes' => ShipmentMode::orderBy('name')->get(),
            'shipment' => $shipment->load(['packages', 'services']),
            'method' => 'PUT',
            'action' => route('shipments.update', $shipment->id),
            'currencyRates' => $currencyRates
        ]);
    }

    public function update(UpdateShipmentRequest $request, Shipment $shipment)
    {
        try {
            DB::transaction(function () use ($request, $shipment) {
                $this->shipmentService->updateShipment($shipment, $request->validated());
                
                // Remove existing packages and services
                $shipment->packages()->delete();
                $shipment->services()->delete();
                
                $this->packageService->processPackages(
                    $request->packages,
                    $shipment,
                    $request->origin_country_id,
                    $request->destination_country_id,
                    $request->mode
                );
                
                if ($request->services) {
                    $this->shipmentService->addServices($shipment, $request->services);
                }
                
                $this->shipmentService->updateTotals($shipment);
            });

            return redirect()->route('shipments.show', $shipment->id)
                ->with('success', 'Shipment updated successfully.');
        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'Error updating shipment: ' . $e->getMessage());
        }
    }

    public function destroy(Shipment $shipment)
    {
        try {
            DB::transaction(function () use ($shipment) {
                $shipment->services()->delete();
                $shipment->packages()->delete();
                $shipment->delete();
            });

            return redirect()->route('shipments.index')
                ->with('success', 'Shipment deleted successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Error deleting shipment: ' . $e->getMessage());
        }
    }

    public function printLabels(Shipment $shipment)
    {
        $shipment->load('packages');
        $pdf = $this->labelService->generateShipmentLabels($shipment);
        
        return $pdf->stream("shipment-labels-{$shipment->id}.pdf");
    }

    public function preview(Request $request)
    {
        $validated = $request->validate([
            'origin_country_id' => 'required',
            'destination_country_id' => 'required',
            'mode' => 'required',
            'category_id' => 'required',
            'packages' => 'required|array',
            'packages.*.length' => 'required|numeric',
            'packages.*.width' => 'required|numeric',
            'packages.*.height' => 'required|numeric',
            'packages.*.weight' => 'required|numeric',
        ]);

        $freight = $this->shipmentService->calculateFreight(
            $validated['packages'],
            $validated['origin_country_id'],
            $validated['destination_country_id'],
            $validated['category_id'],
            $validated['mode']
        );

        $serviceTotal = $request->services 
            ? $this->shipmentService->calculateServiceTotal($request->services)
            : 0;

        return response()->json([
            'freight' => number_format($freight, 2),
            'services_total' => number_format($serviceTotal, 2),
            'total' => number_format($freight + $serviceTotal, 2),
            'currency' => Currency::find($request->charge_currency_id)?->code ?? 'KES'
        ]);
    }

    public function find(FindRouteRequest $request)
    {
        $cacheKey = "route_{$request->origin_country_id}_{$request->destination_country_id}_{$request->category_id}_{$request->mode}";
        
        return Cache::remember($cacheKey, now()->addHours(6), function() use ($request) {
            $shipmentMode = ShipmentMode::where('code', $request->mode)->first();
            
            if (!$shipmentMode) {
                return response()->json(['found' => false]);
            }

            $route = ShipmentRoute::with('currency')
                ->where([
                    'origin_country_id' => $request->origin_country_id,
                    'destination_country_id' => $request->destination_country_id,
                    'category_id' => $request->category_id,
                    'shipment_mode_id' => $shipmentMode->id
                ])
                ->first();

            if (!$route) {
                return response()->json(['found' => false]);
            }

            return response()->json([
                'found' => true,
                'rate_per_unit' => $route->rate_per_unit,
                'flat_rate' => $route->flat_rate,
                'unit_type' => $route->unit_type,
                'currency' => $route->currency->code,
                'exchange_rate' => $route->currency->code === 'KES' 
                    ? 1 
                    : $route->currency->buying_exchange_rate,
                'rates' => ShipmentRoute::getRateBreakdown($route->id),
                'rate_history' => ShipmentRouteHistory::where('route_id', $route->id)
                    ->orderBy('created_at', 'desc')
                    ->limit(5)
                    ->get()
                    ->map(function($h) {
                        return [
                            'date' => $h->created_at->format('M d, Y'),
                            'rate' => $h->rate_per_unit,
                            'currency' => $h->currency,
                            'changed_by' => $h->user->name ?? 'System'
                        ];
                    })
            ]);
        });
    }
}