{{-- resources/views/currencies/currency-form.blade.php --}}
@props([
    'currency' => null, // The currency model (null for create)
    'action' => '',     // Form action URL
    'method' => 'POST', // Form method (POST for create, PUT/PATCH for edit)
])

<form action="{{ $action }}" method="POST" class="needs-validation" novalidate>
    @csrf
    @method($method)

    <div class="row g-3">
        {{-- Currency Code --}}
        <div class="col-md-6 mb-3">
            <label for="code" class="form-label">Currency Code <span class="text-danger">*</span></label>
            <input type="text" 
                   class="form-control @error('code') is-invalid @enderror" 
                   id="code" 
                   name="code" 
                   value="{{ old('code', $currency->code ?? '') }}"
                   required
                   maxlength="3"
                   pattern="[A-Z]{3}"
                   title="3-letter uppercase currency code (e.g. USD, EUR)"
                   @if($currency) readonly @endif>
            <div class="invalid-feedback">
                @error('code') {{ $message }} @else Please provide a valid 3-letter currency code. @enderror
            </div>
            <small class="form-text text-muted">ISO 4217 currency code (e.g. USD, EUR, GBP)</small>
        </div>

        {{-- Currency Name --}}
        <div class="col-md-6 mb-3">
            <label for="name" class="form-label">Currency Name <span class="text-danger">*</span></label>
            <input type="text" 
                   class="form-control @error('name') is-invalid @enderror" 
                   id="name" 
                   name="name" 
                   value="{{ old('name', $currency->name ?? '') }}"
                   required>
            <div class="invalid-feedback">
                @error('name') {{ $message }} @else Please provide the currency name. @enderror
            </div>
        </div>

        {{-- Currency Symbol --}}
        <div class="col-md-6 mb-3">
            <label for="symbol" class="form-label">Currency Symbol</label>
            <input type="text" 
                   class="form-control @error('symbol') is-invalid @enderror" 
                   id="symbol" 
                   name="symbol" 
                   value="{{ old('symbol', $currency->symbol ?? '') }}"
                   maxlength="5">
            <div class="invalid-feedback">
                @error('symbol') {{ $message }} @enderror
            </div>
            <small class="form-text text-muted">Symbol used to represent the currency (e.g. $, €, £)</small>
        </div>

        {{-- Buying Exchange Rate --}}
        <div class="col-md-6 mb-3">
            <label for="buying_exchange_rate" class="form-label">Buying Rate (vs KES) <span class="text-danger">*</span></label>
            <div class="input-group">
                <span class="input-group-text">1 {{ $currency?->code ?? 'NEW' }} =</span>
                <input type="number" 
                       class="form-control @error('buying_exchange_rate') is-invalid @enderror" 
                       id="buying_exchange_rate" 
                       name="buying_exchange_rate" 
                       value="{{ old('buying_exchange_rate', $currency->buying_exchange_rate ?? '') }}"
                       step="0.0001"
                       min="0.0001"
                       required>
                <span class="input-group-text">KES</span>
                <div class="invalid-feedback">
                    @error('buying_exchange_rate') {{ $message }} @else Please provide a valid buying rate. @enderror
                </div>
            </div>
            <small class="form-text text-muted">Rate at which we buy this currency (1 unit = X KES)</small>
        </div>

        {{-- Selling Exchange Rate --}}
        <div class="col-md-6 mb-3">
            <label for="selling_exchange_rate" class="form-label">Selling Rate (vs KES) <span class="text-danger">*</span></label>
            <div class="input-group">
                <span class="input-group-text">1 {{ $currency?->code ?? 'NEW' }} =</span>
                <input type="number" 
                       class="form-control @error('selling_exchange_rate') is-invalid @enderror" 
                       id="selling_exchange_rate" 
                       name="selling_exchange_rate" 
                       value="{{ old('selling_exchange_rate', $currency->selling_exchange_rate ?? '') }}"
                       step="0.0001"
                       min="0.0001"
                       required>
                <span class="input-group-text">KES</span>
                <div class="invalid-feedback">
                    @error('selling_exchange_rate') {{ $message }} @else Please provide a valid selling rate. @enderror
                </div>
            </div>
            <small class="form-text text-muted">Rate at which we sell this currency (1 unit = X KES)</small>
        </div>

        {{-- Service Relationship Warning --}}
        @if($currency && $currency->services()->exists())
        <div class="col-12 alert alert-warning">
            <i class="ri-alert-line"></i> This currency is used by {{ $currency->services()->count() }} services. 
            Changing rates may affect existing pricing.
        </div>
        @endif

        {{-- Form Actions --}}
        <div class="col-12 mt-4">
            <button type="submit" class="btn btn-primary me-2">
                <i class="ri-save-line"></i> {{ $currency ? 'Update' : 'Save' }} Currency
            </button>
            <a href="{{ route('currencies.index') }}" class="btn btn-outline-secondary">
                <i class="ri-arrow-go-back-line"></i> Cancel
            </a>
            
            @if($currency && $currency->code !== 'KES')
                <button type="button" 
                        class="btn btn-danger float-end" 
                        data-bs-toggle="modal" 
                        data-bs-target="#deleteCurrencyModal"
                        @if($currency->services()->exists()) disabled title="Cannot delete - currency is in use by services" @endif>
                    <i class="ri-delete-bin-line"></i> Delete
                </button>
            @endif
        </div>
    </div>
</form>

{{-- Delete Confirmation Modal --}}
@if($currency && $currency->code !== 'KES' && !$currency->services()->exists())
<div class="modal fade" id="deleteCurrencyModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Deletion</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the currency <strong>{{ $currency->code }} - {{ $currency->name }}</strong>?</p>
                <p class="text-danger">This action cannot be undone.</p>
                
                @if($currency->services()->exists())
                <div class="alert alert-danger">
                    <i class="ri-error-warning-line"></i> Warning: This currency is used by {{ $currency->services()->count() }} services.
                </div>
                @endif
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form action="{{ route('currencies.destroy', $currency->id) }}" method="POST">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger" @if($currency->services()->exists()) disabled @endif>
                        <i class="ri-delete-bin-line"></i> Confirm Delete
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>
@endif

@push('scripts')
<script>
// Client-side validation
document.addEventListener('DOMContentLoaded', function() {
    // Validate that selling rate >= buying rate
    const buyingRate = document.getElementById('buying_exchange_rate');
    const sellingRate = document.getElementById('selling_exchange_rate');
    
    function validateRates() {
        if (parseFloat(sellingRate.value) < parseFloat(buyingRate.value)) {
            sellingRate.setCustomValidity('Selling rate must be greater than or equal to buying rate');
        } else {
            sellingRate.setCustomValidity('');
        }
    }
    
    buyingRate.addEventListener('change', validateRates);
    sellingRate.addEventListener('change', validateRates);
    
    // Bootstrap validation
    const forms = document.querySelectorAll('.needs-validation');
    Array.from(forms).forEach(form => {
        form.addEventListener('submit', event => {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            form.classList.add('was-validated');
        }, false);
    });
});
</script>
@endpush