// public/js/suppliers/utils/helpers.js
// Global helpers used across supplier modules.
// Attach to window.SuppliersHelpers

(function (global) {
    class SuppliersHelpers {
        static formatCurrency(amount = 0, currency = 'KES') {
            try {
                return new Intl.NumberFormat('en-KE', {
                    style: 'currency',
                    currency
                }).format(Number(amount));
            } catch (e) {
                return `${currency} ${Number(amount).toFixed(2)}`;
            }
        }

        static formatDate(dateString) {
            if (!dateString) return '-';
            const d = new Date(dateString);
            return d.toLocaleDateString('en-KE', { year: 'numeric', month: 'short', day: 'numeric' });
        }

        static getCsrfToken() {
            const meta = document.querySelector('meta[name="csrf-token"]');
            return meta ? meta.getAttribute('content') : '';
        }

        static debounce(fn, wait = 300) {
            let t;
            return function (...args) {
                clearTimeout(t);
                t = setTimeout(() => fn.apply(this, args), wait);
            };
        }

        static createElementFromHTML(htmlString) {
            const div = document.createElement('div');
            div.innerHTML = htmlString.trim();
            return div.firstChild;
        }

        static showGlobalLoading(show = true) {
            const overlay = document.getElementById('globalLoading');
            if (!overlay) return;
            overlay.style.display = show ? 'block' : 'none';
        }

        /**
         * Generic fetch wrapper that:
         * - injects X-CSRF-TOKEN
         * - returns parsed JSON if possible
         * - throws on non-OK with useful payload
         */
        static async request(url, options = {}) {
            const defaultHeaders = {
                'X-Requested-With': 'XMLHttpRequest'
            };

            const token = SuppliersHelpers.getCsrfToken();
            if (token) defaultHeaders['X-CSRF-TOKEN'] = token;

            options.headers = Object.assign(defaultHeaders, options.headers || {});
            // If formData, do not set content-type (browser will set it)
            if (options.body && !(options.body instanceof FormData)) {
                if (!options.headers['Content-Type'] && !options.headers['content-type']) {
                    options.headers['Content-Type'] = 'application/json';
                }
                // allow body as object
                if (options.headers['Content-Type'] === 'application/json' && typeof options.body !== 'string') {
                    options.body = JSON.stringify(options.body);
                }
            }

            const resp = await fetch(url, options);
            const ct = resp.headers.get('content-type') || '';

            let payload;
            if (ct.includes('application/json')) {
                payload = await resp.json();
            } else {
                payload = await resp.text();
            }

            if (!resp.ok) {
                const err = new Error(payload && payload.message ? payload.message : resp.statusText || 'Request failed');
                err.status = resp.status;
                err.payload = payload;
                throw err;
            }

            return payload;
        }
    }

    global.SuppliersHelpers = SuppliersHelpers;
})(window);
