// public/js/suppliers/modules/vendor-manager.js
// Responsible for initial vendor list rendering, actions (view/edit/delete), and reacting to events.
(function (global) {
    class VendorManager {
        constructor(app) {
            this.app = app;
            this.tableSelector = '#vendors-table'; // fallback table id used by blade
            // If you use other tables, adapt or the module will try to find common ones
        }

        init() {
            // initial load
            document.addEventListener('suppliers:viewChanged', (e) => {
                const view = e.detail.view || this.app.currentVendorType;
                this.loadList(view);
            });

            document.addEventListener('suppliers:requestRefresh', (e) => {
                const view = e?.detail?.view || this.app.currentVendorType;
                this.loadList(view);
            });

            // search results render
            document.addEventListener('suppliers:searchResults', (e) => {
                const { items, view } = e.detail || {};
                this.renderList(items || [], view);
            });

            // when data changes (create/update/delete)
            document.addEventListener('suppliers:dataChanged', (e) => {
                const view = e.detail?.type || this.app.currentVendorType;
                this.loadList(view);
            });

            // API: if the page already includes table data, do not immediately overwrite
            this.loadList(this.app.currentVendorType);
            this.setupDelegatedActions();
        }

        async loadList(view = 'service_providers') {
            // Attempt to load from API routes that should exist
            // for presentation, controllers can return dummy JSON at /api/vendors/service-providers
            const endpoint = view === 'suppliers' ? '/api/vendors/suppliers' : '/api/vendors/service-providers';
            try {
                SuppliersHelpers.showGlobalLoading(true);
                const res = await SuppliersHelpers.request(endpoint, { method: 'GET' });
                // if API returns { data: [...] }
                const items = (res && res.data) ? res.data : (Array.isArray(res) ? res : []);
                this.renderList(items, view);
                // update stats if provided
                if (res && res.stats) {
                    document.dispatchEvent(new CustomEvent('suppliers:statsUpdated', { detail: res.stats }));
                }
            } catch (err) {
                // As fallback: try to use server-rendered HTML already on page; do nothing
                console.warn('VendorManager: failed to load list from API (fallback to page HTML).', err.message || err);
            } finally {
                SuppliersHelpers.showGlobalLoading(false);
            }
        }

        renderList(items, view = 'service_providers') {
            // find table body to render
            const table = document.querySelector(this.tableSelector) || document.querySelector('#service-providers-section table') || document.querySelector('#vendors-table');
            if (!table) {
                // No table found; nothing to render
                return;
            }
            const tbody = table.querySelector('tbody') || (function () {
                const tb = document.createElement('tbody');
                table.appendChild(tb);
                return tb;
            })();

            tbody.innerHTML = '';
            if (!items || !items.length) {
                tbody.innerHTML = `<tr><td colspan="5" class="text-center text-muted">No ${view.replace('_', ' ')} found</td></tr>`;
                return;
            }

            items.forEach((item, index) => {
                const tr = document.createElement('tr');
                const actionButtons = `
                    <button class="btn btn-sm btn-info me-1" data-action="view" data-id="${item.id}"><i class="ri-eye-line"></i></button>
                    <button class="btn btn-sm btn-warning me-1" data-action="edit" data-id="${item.id}"><i class="ri-edit-2-line"></i></button>
                    <button class="btn btn-sm btn-danger" data-action="delete" data-id="${item.id}"><i class="ri-delete-bin-line"></i></button>
                `;
                tr.innerHTML = `
                    <td>${index + 1}</td>
                    <td><a href="/service-providers/${item.id}" class="vendor-link" data-id="${item.id}">${item.name || item.title || '—'}</a></td>
                    <td>${item.type || (view === 'suppliers' ? 'Supplier' : 'Service Provider')}</td>
                    <td>${item.status ? `<span class="badge ${item.status === 'active' ? 'bg-success' : 'bg-danger'}">${item.status}</span>` : '-'}</td>
                    <td>${actionButtons}</td>
                `;
                tbody.appendChild(tr);
            });
        }

        setupDelegatedActions() {
            document.addEventListener('click', async (e) => {
                const btn = e.target.closest('button[data-action], a.vendor-link');
                if (!btn) return;
                // link to view
                if (btn.matches('a.vendor-link')) {
                    // allow navigation naturally
                    return;
                }
                const action = btn.getAttribute('data-action');
                const id = btn.getAttribute('data-id');
                if (!action || !id) return;
                switch (action) {
                    case 'view':
                        window.location.href = `/service-providers/${id}`;
                        break;
                    case 'edit':
                        this.openEditModal(id);
                        break;
                    case 'delete':
                        await this.deleteVendor(id);
                        break;
                }
            });
        }

        async openEditModal(id) {
            try {
                SuppliersHelpers.showGlobalLoading(true);
                const data = await SuppliersHelpers.request(`/service-providers/${id}/edit-data`, { method: 'GET' });
                // populate modal if it exists
                const modalEl = document.getElementById('serviceProviderModal') || document.getElementById('vendor-modal');
                if (!modalEl) {
                    this.app.showAlert('Edit modal not found on page', 'warning');
                    return;
                }
                const form = modalEl.querySelector('form');
                if (!form) {
                    this.app.showAlert('Edit form not found in modal', 'warning');
                    return;
                }
                // Fill simple fields
                Object.keys(data || {}).forEach(k => {
                    const input = form.querySelector(`[name="${k}"]`);
                    if (input) {
                        input.value = data[k];
                    }
                });
                const modalManager = this.app.getModule('modalManager');
                modalManager && modalManager.open(modalEl.id);
            } catch (err) {
                this.app.showAlert('Failed to load vendor data: ' + (err.message || ''), 'error');
            } finally {
                SuppliersHelpers.showGlobalLoading(false);
            }
        }

        async deleteVendor(id) {
            if (!confirm('Are you sure you want to delete this vendor?')) return;
            try {
                SuppliersHelpers.showGlobalLoading(true);
                await SuppliersHelpers.request(`/service-providers/${id}`, { method: 'DELETE' });
                this.app.showAlert('Vendor deleted', 'success');
                document.dispatchEvent(new CustomEvent('suppliers:dataChanged', { detail: { type: this.app.currentVendorType } }));
            } catch (err) {
                this.app.showAlert('Delete failed: ' + (err.message || ''), 'error');
            } finally {
                SuppliersHelpers.showGlobalLoading(false);
            }
        }
    }

    global.VendorManager = VendorManager;
})(window);
