// public/js/suppliers/modules/search-manager.js
// Live search that calls server API or triggers client-side filtering.
(function (global) {
    class SearchManager {
        constructor(app) {
            this.app = app;
            this.debounced = SuppliersHelpers.debounce(this.performSearch.bind(this), 300);
        }

        init() {
            this.bind();
        }

        bind() {
            document.querySelectorAll('.vendor-search').forEach(input => {
                input.addEventListener('input', (e) => {
                    const q = e.target.value.trim();
                    this.debounced(q);
                });
            });

            // delegate search in-card (search box inside card-body)
            document.addEventListener('input', (e) => {
                const el = e.target.closest('.card .search-box input');
                if (!el) return;
                this.debounced(el.value.trim());
            });
        }

        async performSearch(q) {
            const view = this.app.currentVendorType || 'service_providers';
            if (!q) {
                // refresh list
                document.dispatchEvent(new CustomEvent('suppliers:searchCleared', { detail: { view } }));
                document.dispatchEvent(new CustomEvent('suppliers:requestRefresh', { detail: { view } }));
                return;
            }

            try {
                SuppliersHelpers.showGlobalLoading(true);
                // call search endpoint - controller should support this
                const url = `/service-providers/api/search?q=${encodeURIComponent(q)}&type=${encodeURIComponent(view)}`;
                const res = await SuppliersHelpers.request(url, { method: 'GET' });
                // Expect res.data array or res.html fragment
                if (res.html) {
                    // insert into table wrapper if provided (controller must return proper HTML)
                    const wrapper = document.querySelector('#service-providers-section .card-body') || document.querySelector('#providers .card-body');
                    if (wrapper) wrapper.innerHTML = res.html;
                } else if (res.data) {
                    // publish event for VendorManager to render
                    document.dispatchEvent(new CustomEvent('suppliers:searchResults', { detail: { items: res.data, view } }));
                }
            } catch (err) {
                this.app.showAlert('Search failed: ' + (err.message || ''), 'error');
            } finally {
                SuppliersHelpers.showGlobalLoading(false);
            }
        }
    }

    global.SearchManager = SearchManager;
})(window);
