// public/js/suppliers/modules/modal-manager.js
// Modal helper: initializes bootstrap modals and provides helpers for dynamic route rows, weight-divisor UI, and AJAX submit.
(function (global) {
    class ModalManager {
        constructor(app) {
            this.app = app;
            this.modals = {};
        }

        init() {
            this.initBootstrapModals();
            this.bindGlobalClicks();
        }

        initBootstrapModals() {
            document.querySelectorAll('.modal').forEach(el => {
                try {
                    this.modals[el.id] = new bootstrap.Modal(el, { backdrop: 'static' });
                    el.addEventListener('hidden.bs.modal', () => {
                        const form = el.querySelector('form');
                        if (form) {
                            form.reset();
                            form.querySelectorAll('.is-invalid').forEach(i => i.classList.remove('is-invalid'));
                            const err = form.querySelector('.form-errors');
                            if (err) err.innerHTML = '';
                        }
                    });
                } catch (err) {
                    // bootstrap not present or modal invalid
                }
            });
        }

        bindGlobalClicks() {
            // add route-row
            document.addEventListener('click', (e) => {
                const btn = e.target.closest('[data-action="add-route-row"]');
                if (btn) {
                    e.preventDefault();
                    const containerSelector = btn.getAttribute('data-target') || '.routes-container';
                    const container = document.querySelector(containerSelector);
                    if (container) this.addRouteRow(container);
                }
            });

            // remove route-row
            document.addEventListener('click', (e) => {
                const btn = e.target.closest('[data-action="remove-route-row"]');
                if (btn) {
                    e.preventDefault();
                    const row = btn.closest('.route-row');
                    row && row.remove();
                }
            });

            // show divisor on weight-basis change
            document.addEventListener('change', (e) => {
                const sel = e.target.closest('.weight-basis-select');
                if (sel) this.onWeightBasisChange(sel);
            });

            // generic AJAX form submit for forms with .ajax-form
            document.addEventListener('submit', (e) => {
                const form = e.target;
                if (form && form.classList.contains('ajax-form')) {
                    e.preventDefault();
                    this.submitForm(form);
                }
            });
        }

        addRouteRow(container) {
            const rows = container.querySelectorAll('.route-row');
            const idx = rows.length;
            const html = `
                <div class="row g-2 mb-2 route-row">
                    <div class="col-md-3"><input name="routes[${idx}][from]" class="form-control" placeholder="From"></div>
                    <div class="col-md-3"><input name="routes[${idx}][to]" class="form-control" placeholder="To"></div>
                    <div class="col-md-2"><input name="routes[${idx}][flat_rate]" type="number" class="form-control" placeholder="Flat rate"></div>
                    <div class="col-md-2">
                        <select name="routes[${idx}][weight_basis]" class="form-select weight-basis-select">
                            <option value="gross">Gross</option>
                            <option value="net">Net</option>
                            <option value="volumetric">Volumetric (air - 6000)</option>
                            <option value="cbm">CBM (1,000,000)</option>
                            <option value="courier">Courier (50,000)</option>
                            <option value="cuft">CuFt (50,000)</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <button class="btn btn-danger w-100" data-action="remove-route-row"><i class="ri-delete-bin-line"></i></button>
                    </div>
                </div>
            `;
            container.insertAdjacentHTML('beforeend', html);
        }

        onWeightBasisChange(select) {
            const divisorMap = {
                volumetric: 6000,
                cbm: 1000000,
                courier: 50000,
                cuft: 50000
            };
            const val = select.value;
            const row = select.closest('.route-row');
            if (!row) return;
            let disp = row.querySelector('.divisor-display');
            if (!disp) {
                disp = document.createElement('div');
                disp.className = 'form-text divisor-display';
                select.parentNode.appendChild(disp);
            }
            if (divisorMap[val]) {
                disp.textContent = `Divisor: ${divisorMap[val].toLocaleString()}`;
            } else {
                disp.textContent = '';
            }
        }

        async submitForm(form) {
            const url = form.action || window.location.href;
            const method = (form.querySelector('input[name="_method"]')?.value || form.method || 'POST').toUpperCase();
            const isFormData = form.enctype === 'multipart/form-data';
            let body;
            if (isFormData) {
                body = new FormData(form);
            } else {
                // convert form fields to JSON-friendly object
                const fd = new FormData(form);
                const obj = {};
                for (const [k, v] of fd.entries()) {
                    // simple handling: arrays via [] can be supported later
                    obj[k] = v;
                }
                body = JSON.stringify(obj);
            }

            const headers = isFormData ? {} : { 'Content-Type': 'application/json' };

            const submitBtn = form.querySelector('button[type="submit"]');
            try {
                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.dataset.original = submitBtn.innerHTML;
                    submitBtn.innerHTML = `<span class="spinner-border spinner-border-sm me-2"></span>Saving...`;
                }
                SuppliersHelpers.showGlobalLoading(true);
                const res = await SuppliersHelpers.request(url, { method, body, headers });
                if (res && res.success) {
                    this.app.showAlert(res.message || 'Saved', 'success');
                    // close modal if exists
                    const modalEl = form.closest('.modal');
                    if (modalEl && this.modals[modalEl.id]) this.modals[modalEl.id].hide();
                    // broadcast event so list can refresh
                    document.dispatchEvent(new CustomEvent('suppliers:dataChanged', { detail: { type: this.app.currentVendorType } }));
                    // small delay for UX
                    setTimeout(() => window.location.reload(), 600);
                } else {
                    // show errors
                    this.showFormErrors(form, res.errors || {});
                }
            } catch (err) {
                this.app.showAlert(err.message || 'Save failed', 'error');
                if (err.payload && err.payload.errors) {
                    this.showFormErrors(form, err.payload.errors);
                }
            } finally {
                if (submitBtn) {
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = submitBtn.dataset.original || submitBtn.innerHTML;
                }
                SuppliersHelpers.showGlobalLoading(false);
            }
        }

        showFormErrors(form, errors = {}) {
            const errBox = form.querySelector('.form-errors') || (function () {
                const div = document.createElement('div');
                div.className = 'form-errors text-danger mb-2';
                form.prepend(div);
                return div;
            })();
            errBox.innerHTML = '';
            Object.values(errors).forEach(arr => {
                (arr || []).forEach(msg => {
                    const p = document.createElement('div');
                    p.textContent = msg;
                    errBox.appendChild(p);
                });
            });
            Object.keys(errors).forEach(k => {
                const el = form.querySelector(`[name="${k}"]`);
                if (el) el.classList.add('is-invalid');
            });
        }

        open(modalId) {
            this.modals[modalId]?.show();
        }

        close(modalId) {
            this.modals[modalId]?.hide();
        }
    }

    global.ModalManager = ModalManager;
})(window);
